import type { GraphApi } from '@sage/wh-system-api';
import * as ui from '@sage/xtrem-ui';
import { BarcodeManagementService } from './barcode-management-service';
import {
    AsyncCompositeAllowed,
    BarcodeManagerSupportedFields,
    DictionaryDataGeneric,
    DictionaryGenericFieldSupported,
} from './screen-management';

export class ManagementServiceComposite extends BarcodeManagementService {
    /**
     * This Method build instance with retrieving all data
     * @param pageInstance : current page
     * @param dictionaryFieldSupported : page fields supported
     * @param dictionaryDataComposites : extracted composite data, may be send to storage
     * @param checkCompositeDataAllowed? : optional client callback to check composite data code before to dispatch them
     */

    // eslint-disable-next-line require-await
    public override async initialize(
        pageInstance: ui.Page<GraphApi>,
        dictionaryFieldSupported: DictionaryGenericFieldSupported = {} as DictionaryGenericFieldSupported,
        dictionaryDataComposites: DictionaryDataGeneric = {} as DictionaryDataGeneric,
        checkCompositeDataAllowed?: AsyncCompositeAllowed,
    ): Promise<boolean> {
        return super.initialize(
            pageInstance,
            this._getDictionaryFieldSupported(dictionaryFieldSupported),
            dictionaryDataComposites,
            checkCompositeDataAllowed,
        );
    }

    /**
     * Check if raw data is a composite data
     * @param rawData
     * @returns
     */
    public override isCompositeData(rawData: any): boolean {
        // prefix to exclude GS1 composite data
        const groupSeparator = String.fromCharCode(0x1d); // GS1 Group Separator
        const compositeCodePrefix = new RegExp(
            `^(\\][A-Za-z]\\d{3,5}|${groupSeparator.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\d{2,4})`,
        );
        return (
            rawData &&
            typeof rawData === 'string' &&
            rawData.length >= Math.max(4, Number(this.barcodeManagementConfiguration?.recordSize)) &&
            !Number(rawData.match(compositeCodePrefix)?.length ?? 0)
        );
    }

    /**
     * Called when field update is in progress
     * @param pageInstance current page
     * @param currentField reference of field to updating
     * @param rawData received raw data
     * @param appendCompositeData true when is necessary to appending current date to previous set
     * @return true when is not necessary to continue and affect a value : composite value has been processed
     */
    public override scan(
        pageInstance: ui.Page<GraphApi>,
        currentField: BarcodeManagerSupportedFields,
        rawData: any,
        appendCompositeData: boolean = false,
    ): Promise<boolean> {
        return super.scan(pageInstance, currentField, rawData, appendCompositeData);
    }

    /**
     * Update screen fields supported.
     * field mapping was done after
     * @param pageInstance current page
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    public override setScreenFieldSupported(
        pageInstance: ui.Page<GraphApi>,
        dictionaryFieldSupported: DictionaryGenericFieldSupported,
    ): Promise<boolean> {
        return super.setScreenFieldSupported(pageInstance, this._getDictionaryFieldSupported(dictionaryFieldSupported));
    }

    /**
     * Load composite data from storage and erase them
     * @param _dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when some data has been loaded
     */
    private _getDictionaryFieldSupported(
        _dictionaryFieldSupported: DictionaryGenericFieldSupported,
    ): DictionaryGenericFieldSupported {
        return this.barcodeManagementConfiguration?.dictionaryFieldSupported ?? ({} as DictionaryGenericFieldSupported);
    }
}
