import { SystemError } from '@sage/xtrem-shared';
import * as ui from '@sage/xtrem-ui';
import { GenericPrintPayloadMapping } from './generic-print-payload-mapping';

export async function genericPrintReport(
    pageInstance: ui.Page,
    reportName: string,
    language: string,
    destination: string,
    graphQlPayload: any,
): Promise<number> {
    let result: number;

    if ((reportName || '') === '') {
        throw new SystemError('Report name is mandatory');
    }

    if ((destination || '') === '') {
        throw new SystemError('Destination is mandatory');
    }

    // Custom parameters has object
    if (graphQlPayload == null || typeof graphQlPayload !== 'object' || Object.keys(graphQlPayload).length === 0) {
        throw new SystemError('Custom parameters is mandatory');
    }

    ui.console.debug(`Custom payload in input : ${JSON.stringify(graphQlPayload)}`);

    const _customParameters = GenericPrintPayloadMapping.buildGraphQLPayload(graphQlPayload);

    ui.console.debug(`Prepared custom payload : ${JSON.stringify(_customParameters)}`);

    /**
     * Explanation of errorCode cases :
     *  n = Error code number.
     *  0 = Done.
     * -1 = Result of mutation not return expected errorCode message.
     * -2 = Failure without error when calling mutation (unchanged value).
     */

    // Populate required arguments

    const _mutationArgs = {
        reportName,
        language,
        destination,
        parametersNames: _customParameters.parametersNames,
        parametersValues: _customParameters.parametersValues,
        errorCode: -2,
    };

    ui.console.debug(`Prepared mutation arguments : ${JSON.stringify(_mutationArgs)}`);

    try {
        result =
            (
                await pageInstance.$.graph
                    .node('@sage/wh-system/GenericPrintReport')
                    .mutations.printReport(
                        {
                            reportName: false,
                            language: false,
                            destination: false,
                            parametersNames: false,
                            parametersValues: false,
                            errorCode: true,
                        },
                        {
                            parameters: _mutationArgs,
                        },
                    )
                    .execute()
            )?.errorCode ?? -1;
    } catch (error) {
        throw new SystemError(error.message, error);
    }
    return result;
}
