"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BarcodeParserComposite = void 0;
const typesLib = require("@sage/xtrem-decimal");
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const barcode_parser_abstract_1 = require("./barcode-parser-abstract");
const barcode_parser_configuration_1 = require("./barcode-parser-configuration");
const parsed_generic_element_1 = require("./parsed-generic-element");
class BarcodeParserComposite extends barcode_parser_abstract_1.AbstractBarcodeParser {
    parse(barCode) {
        try {
            const _parserConfiguration = this.parserConfiguration;
            if (!_parserConfiguration) {
                throw new Error('Parser configuration is missing');
            }
            const _recordSize = _parserConfiguration.recordSize;
            if (!_recordSize) {
                throw new Error('Record size is missing');
            }
            if (typesLib.lt(barCode.length, _recordSize)) {
                throw new Error(`Barcode length is not valid, expected minimum ${_recordSize} but got ${barCode.length}`);
            }
            const _sizeDate = (0, barcode_parser_configuration_1.getDateFormatSize)(_parserConfiguration.dateFormat);
            const _dateFormat = (0, barcode_parser_configuration_1.getDateFormat)(_parserConfiguration.dateFormat);
            if (typesLib.strictNe(_dateFormat.length, _sizeDate)) {
                throw new Error(`Date format size  ${_dateFormat} is not valid`);
            }
            let _parsedCodeItems = [];
            // Use string comparison to avoid enum comparison issues
            const compositeFormat = String(_parserConfiguration.compositeFormat);
            if (typesLib.strictEq(compositeFormat, 'fixed')) {
                _parsedCodeItems = this._parseFixed(barCode, _dateFormat, _sizeDate);
            }
            else if (typesLib.strictEq(compositeFormat, 'delimited')) {
                _parsedCodeItems = this._parseDelimited(barCode, _dateFormat, _sizeDate);
            }
            else {
                throw new Error(`Composite format ${_parserConfiguration.compositeFormat} is not supported`);
            }
            return {
                codeName: _parserConfiguration.compositeCode ?? _parserConfiguration.localizedDescription ?? '',
                parsedCodeItems: _parsedCodeItems,
            };
        }
        catch (error) {
            throw new xtrem_shared_1.SystemError(`Parser error :\n${error}`, error);
        }
    }
    /*
     * Parse the fixed length barcode
     * @param barCode: string - the barcode to parse
     * @param dateFormat: string - the date format
     * @param sizeDate: number - the size of the date
     * @returns ParsedGenericElement[] - the parsed elements of the barcode
     */
    _parseFixed(barCode, dateFormat, sizeDate) {
        const _parsedCodeItems = [];
        const _parserConfiguration = this.parserConfiguration;
        let _startPosition = 0;
        if (!_parserConfiguration) {
            return [];
        }
        const { isBlankDeletion } = _parserConfiguration;
        _parserConfiguration?.lines.forEach(line => {
            const _length = line.length || 0;
            const _fieldKey = line.fieldKey;
            if (typesLib.strictEq(_length, 0)) {
                throw new Error(`Length is missing for field ${_fieldKey}`);
            }
            const _isDateField = typesLib.strictEq(line.fieldDataType, 'date');
            if (_isDateField && typesLib.lt(_length, sizeDate)) {
                throw new Error(`Field size ${_fieldKey} is not valid, expected minimum ${sizeDate} but got ${_length}`);
            }
            const _fieldSize = _isDateField ? typesLib.min(sizeDate, _length) : _length;
            const _endPosition = typesLib.add(_startPosition, _length);
            const _value = barCode.substring(_startPosition, typesLib.add(_startPosition, _fieldSize)) || '';
            _startPosition = _endPosition;
            _parsedCodeItems.push(...this._getParsedElement(line, isBlankDeletion ? _value.trim() : _value, dateFormat, sizeDate));
        });
        return _parsedCodeItems;
    }
    /*
     * Parse the delimited barcode
     * @param barCode: string - the barcode to parse
     * @param _sizeDate: number - the size of the date
     *   @returns ParsedGenericElement[] - the parsed elements of the barcode
     */
    _parseDelimited(barCode, dateFormat, sizeDate) {
        const _parsedCodeItems = [];
        const _parserConfiguration = this.parserConfiguration;
        if (!_parserConfiguration) {
            return [];
        }
        const { isBlankDeletion } = _parserConfiguration;
        const _fields = barCode.split(_parserConfiguration.fieldSeparator);
        if (typesLib.lt(_fields.length, _parserConfiguration.lines.length)) {
            throw new Error(`Barcode fields count is not valid, expected ${_parserConfiguration.lines.length} but got ${_fields.length}`);
        }
        for (let index = 0; typesLib.lt(index, _parserConfiguration.lines.length); index = typesLib.add(index, 1)) {
            const _line = _parserConfiguration.lines[index];
            const _isDateField = typesLib.strictEq(_line.fieldDataType, 'date');
            const _value = (isBlankDeletion || _isDateField ? _fields[index]?.trim() : _fields[index]) || '';
            if (_isDateField && typesLib.lt(_value.length, sizeDate)) {
                if (_value.length) {
                    throw new Error(`Field size ${_line.fieldKey} is not valid, expected minimum ${sizeDate} but got ${_value.length}`);
                }
                // When the date is empty, we don't throw an error but we return an empty date
                const _elementType = (0, parsed_generic_element_1.getElementType)(_line.fieldDataType);
                _parsedCodeItems.push(new parsed_generic_element_1.ParsedGenericElement(_line.fieldKey, _line.fieldKey, _elementType));
                if (_line.fieldAlias) {
                    _parsedCodeItems.push(new parsed_generic_element_1.ParsedGenericElement(_line.fieldAlias, _line.fieldAlias, _elementType));
                }
            }
            else {
                _parsedCodeItems.push(...this._getParsedElement(_line, _value, dateFormat, sizeDate));
            }
        }
        return _parsedCodeItems;
    }
    /*
     * Get the element type
     * @param line: ParserConfigurationLine - the line to parse
     * @param value: string - the value to parse
     * @param dateFormat: string - the date format
     * @param sizeDate: number - the size of the date
     * @returns ParsedGenericElement - the parsed element
     */
    _getParsedElement(line, value, dateFormat, sizeDate) {
        const _elementType = (0, parsed_generic_element_1.getElementType)(line.fieldDataType);
        const _parsedElement = new parsed_generic_element_1.ParsedGenericElement(line.fieldKey, line.fieldKey, _elementType);
        let _parsedAliasElement;
        switch (_elementType) {
            case parsed_generic_element_1.ElementTypeEnum.string:
                _parsedElement.data = value;
                break;
            case parsed_generic_element_1.ElementTypeEnum.number:
                _parsedElement.data = BarcodeParserComposite._parseNumber(line.fieldKey, value, this.parserConfiguration?.decimalSeparator);
                break;
            case parsed_generic_element_1.ElementTypeEnum.date:
                _parsedElement.data = BarcodeParserComposite._parseDate(line.fieldKey, value, dateFormat, sizeDate);
                break;
            default:
                throw new Error(`Element type ${_elementType} is not supported`);
        }
        if (line.fieldAlias) {
            _parsedAliasElement = new parsed_generic_element_1.ParsedGenericElement(line.fieldAlias, line.fieldAlias, _elementType);
            _parsedAliasElement.data = _parsedElement.data;
        }
        return _parsedAliasElement ? [_parsedElement, _parsedAliasElement] : [_parsedElement];
    }
    /**
     * Parse the date
     * @param fieldKey: string - the field key
     * @param value: string - the value to parse
     * @param dateFormat: string - the date format
     * @param sizeDate: number - the size of the date
     * @returns string - the parsed date
     */
    static _parseDate(fieldKey, value, dateFormat, sizeDate) {
        const dateToFormat = value.slice(0, sizeDate);
        if (!dateToFormat.length) {
            throw new Error(`Date  ${fieldKey} is empty`);
        }
        if (typesLib.strictNe(dateToFormat.length, sizeDate)) {
            throw new Error(`Date size ${fieldKey} is not valid, expected ${sizeDate} but got ${dateToFormat.length}`);
        }
        return xtrem_date_time_1.date.parse(dateToFormat, undefined, dateFormat);
    }
    /**
     * Parse the number
     * @param fieldKey: string - the field key
     * @param value: string - the value to parse
     * @param decimalSeparator: string - the decimal separator
     * @returns number - the parsed number
     */
    static _parseNumber(fieldKey, value, decimalSeparator) {
        let _value = value;
        if (decimalSeparator && typesLib.strictNe(decimalSeparator, '.')) {
            _value = value.replace(decimalSeparator, '.');
        }
        const parsedNumber = parseFloat(_value);
        if (Number.isNaN(parsedNumber)) {
            throw new Error(`Number format ${fieldKey} is not valid, expected number but got ${value}`);
        }
        return parsedNumber;
    }
}
exports.BarcodeParserComposite = BarcodeParserComposite;
//# sourceMappingURL=barcode-parser-composite.js.map