"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SupportManagementService = void 0;
const ui = require("@sage/xtrem-ui");
const barcode_management_service_factory_1 = require("./barcode-management-service-factory");
const management_service_gs1_1 = require("./management-service-gs1");
const defaultManagementServiceKey = 'DEFAULT_MOBILE_COMPOSITE_DATA_KEY';
class SupportManagementService extends ui.Page {
    /**
     *  Constructor of control manager GS 1
     * @param typeOfServiceBarCode type of service barcode
     */
    constructor() {
        super();
        /**
         * Internal only
         */
        this._factory = new barcode_management_service_factory_1.BarcodeManagementServiceFactory();
        /**
         * Used only for disable scan operation (service must stay active)
         */
        /** @internal */
        this._disableBarcodeService = false;
        this._factory = new barcode_management_service_factory_1.BarcodeManagementServiceFactory();
        this._controlManager = new management_service_gs1_1.ManagementServiceGs1(defaultManagementServiceKey);
    }
    /**
     * Create a new instance of service barcode
     * @param codeFlow optional flow code
     */
    /** @internal */
    _createService(codeFlow) {
        return this._factory.createService(this, defaultManagementServiceKey, codeFlow);
    }
    /**
     * Initialize ControlManager :
     * Depending on the type of manager used (not Gs1), the dictionary and checkCompositeDataAllowed parameters
     * may be ignored because the factory retrieves the configuration to know whether or not to use them.
     * The flow code is mandatory in order to load the configuration, except in the case of Gs1.
     *
     * @param site current site
     * @param compositePageKey key for storage operation
     * @param codeFlow optional flow code (mandatory for non-Gs1)
     * @param dictionaryFieldSupported fields screen to manage and control
     * @param dictionaryDataComposites? optional composite data
     * @param checkCompositeDataAllowed? optional client callback to check composite data code before to dispatch them
     * @returns true when ControlManager is usable
     */
    /** @internal */
    async _createAndInitControlManager(site, compositePageKey, codeFlow, dictionaryFieldSupported, dictionaryDataComposites, checkCompositeDataAllowed) {
        /**
         * Create and initialize manager for a given service :
         *
         */
        this._controlManager = await this._createService(codeFlow);
        // Including erroneous field misspelled dateTestFake instead dateTest
        if (site && compositePageKey) {
            if (await this._controlManager
                .initialize(this, dictionaryFieldSupported, dictionaryDataComposites, checkCompositeDataAllowed)
                .catch(() => false)) {
                return true;
            }
            // Error detected, abort controller
            ui.console.error(`Service initialization failure for site ${site}`);
        }
        // Initialization failure : reset to default instance (gs1)
        this._controlManager = await this._createService();
        return false;
    }
    /**
     * Remove composite data
     */
    /** @protected */
    clearCompositeData() {
        this._controlManager.clearCompositeData();
    }
    /**
     * Remove composite data from storage
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    /** @protected */
    clearCompositeDataStorage(storageKey) {
        this._controlManager.clearCompositeDataStorage(this, storageKey);
    }
    /**
     * Remove all composite and storage data
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    /** @protected */
    clearAllCompositeDataAndStorage(storageKey) {
        this._controlManager.clearAllCompositeDataAndStorage(this, storageKey);
    }
    /**
     * Get current control manager instance
     * @returns current instance of control manager
     */
    /** @protected */
    get controlManager() {
        return this._controlManager;
    }
    /**
     * Create and initialize manager for a given service :
     * Depending on the type of manager used (not Gs1), the dictionary and checkCompositeDataAllowed parameters
     * may be ignored because the factory retrieves the configuration to know whether or not to use them.
     * The flow code is mandatory in order to load the configuration, except in the case of Gs1.
     *
     * @param site current site
     * @param compositePageKey key for storage operation
     * @param codeFlow optional flow code (mandatory for non-Gs1)
     * @param dictionaryFieldSupported fields screen to manage and control
     * @param dictionaryDataComposites? optional composite data
     * @param checkCompositeDataAllowed? optional client callback to check composite data code before to dispatch them
     * @return false when service has not properly initialized (unavailable)
     */
    /** @protected */
    async createAndInitServiceManagement(site, compositePageKey, codeFlow, dictionaryFieldSupported, dictionaryDataComposites, checkCompositeDataAllowed) {
        if (!site ||
            !(await this._createAndInitControlManager(site, compositePageKey, codeFlow, dictionaryFieldSupported, dictionaryDataComposites, checkCompositeDataAllowed))) {
            ui.console.error(`Unable to initialize service control manager for site ${site}`);
            return false;
        }
        return true;
    }
    /**
     * Disable or enable service (default : enable) :
     * Disabling clearing all composite data
     */
    /** @protected */
    set disableBarcodeService(disableServiceBarcode) {
        this._disableBarcodeService = disableServiceBarcode;
        if (this._disableBarcodeService) {
            this._controlManager.clearCompositeData();
        }
    }
    /**
     * Protected only
     */
    /**
     * Return current state of service
     */
    /** @protected */
    get isBarcodeServiceDisabled() {
        return this._disableBarcodeService;
    }
    /**
     * Load composite data from storage and erase them
     * @returns true when some data has been loaded
     */
    /** @protected */
    loadCompositeData() {
        return this._controlManager.loadCompositeData(this);
    }
    /**
     * Save composite data, zero elements allowed
     * @returns true when operation has performed
     */
    /** @protected */
    saveCompositeData() {
        return this._controlManager.saveCompositeData(this);
    }
    /**
     * Update screen fields supported (only when exists Gs1 parameters).
     * mapping field has been performed after
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    setScreenFieldSupported(dictionaryFieldSupported) {
        return Promise.resolve(this._controlManager.setScreenFieldSupported(this, dictionaryFieldSupported));
    }
    /**
     * call scan evaluator only when manager available and rawData has minimum requirement
     * @param mainField current screen field reference
     * @param rawData data to parse
     * @return false when is not a bar code composite
     */
    /** @protected */
    scan(mainField, rawData) {
        // class must be created, initialized and not busy, data must be started by a numeric value,
        // minimum size 2 to 4 numerics or prefix ]<alphabetic><numeric>
        if (!(this._disableBarcodeService ?? false) && this._controlManager.isCompositeData(rawData)) {
            return this._controlManager.scan(this, mainField, rawData).catch(() => false);
        }
        return Promise.resolve(false);
    }
    /**
     * Return type of service barcode
     * @returns type of service barcode
     *
     */
    get typeOfServiceBarcode() {
        return this._controlManager.typeOfServiceBarcode ?? 'gs1';
    }
}
exports.SupportManagementService = SupportManagementService;
//# sourceMappingURL=support-management-service.js.map