"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BarcodeManagementService = void 0;
const typesLib = require("@sage/xtrem-decimal");
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const ui = require("@sage/xtrem-ui");
const barcode_parser_factory_1 = require("../shared-functions/barcode-parser-factory");
var SetFieldResult;
(function (SetFieldResult) {
    SetFieldResult["error"] = "error";
    SetFieldResult["noData"] = "noData";
    SetFieldResult["unchanged"] = "unchanged";
    SetFieldResult["updated"] = "updated";
})(SetFieldResult || (SetFieldResult = {}));
class BarcodeManagementService {
    /**
     * Initialize permanent data for page and parser
     * @param _storageKey                       : storage key used for this page
     * @param _barcodeManagementConfiguration   : optional configuration
     */
    constructor(_storageKey, _barcodeManagementConfiguration) {
        this._storageKey = _storageKey;
        this._barcodeManagementConfiguration = _barcodeManagementConfiguration;
        // Private data area
        /**
         *  Store all codes GS 1 supported by access code (for speedup decoding)
         */
        /** @internal */
        this._screenFieldsSupported = [];
        /** @internal */
        this._dictionarySegment = {};
        /** @internal */
        this._screenFieldsMapped = [];
        /** @internal */
        this._dictionaryDataComposites = {};
        /** @internal */
        this._abortDispatch = false;
        /** @internal */
        this._dispatchInProgress = false;
        /** @internal */
        this._initializationInProgress = false;
        const _factory = new barcode_parser_factory_1.BarcodeParserFactory(this._barcodeManagementConfiguration?.parserConfiguration);
        this._parseBarCode = _factory.createParser();
    }
    /**
     * Request to abort dispatch without error (end of scan) :
     * This is used only when onChange() has performed some operations,
     * ie : switching page... except when initialization is in progress
     * @param {function}    optional callback to be executed approx. 100ms after end of dispatching.
     * @returns {boolean}   true when request has be set
     */
    abortDispatch(postActionAbortDispatch) {
        this._abortDispatch = false;
        this._postActionAbortDispatch = undefined;
        if (this._dispatchInProgress) {
            this._abortDispatch = true;
            if (!this.isInitializationInProgress) {
                this._postActionAbortDispatch = postActionAbortDispatch;
            }
        }
        return this.isAbortDispatchInProgress;
    }
    /**
     * Get barcode management configuration
     *  @returns barcode management configuration
     */
    get barcodeManagementConfiguration() {
        return this._barcodeManagementConfiguration;
    }
    /**
     * Remove all composite and storage data
     * @param pageInstance current page
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    clearAllCompositeDataAndStorage(pageInstance, storageKey) {
        this.clearCompositeData();
        this.clearCompositeDataStorage(pageInstance, storageKey);
    }
    /**
     * Remove composite data
     */
    clearCompositeData() {
        this._dictionaryDataComposites = {};
    }
    /**
     * Remove composite data from storage
     * @param pageInstance current page
     * @param storageKey optional, for overriding current service key (use with caution)
     */
    clearCompositeDataStorage(pageInstance, storageKey) {
        this._removeCompositeDataStorage(pageInstance, storageKey);
    }
    /*
     * Get scanned field ID when processing field during scanning only
     * @returns scanned field id
     */
    get scannedFieldId() {
        return this._scannedFieldId;
    }
    /**
     * This Method build instance with retrieving all data
     * @param pageInstance : current page
     * @param dictionaryFieldSupported : page fields supported
     * @param dictionaryDataComposites : extracted composite data, may be send to storage
     * @param checkCompositeDataAllowed? : optional client callback to check composite data code before to dispatch them
     */
    async initialize(pageInstance, dictionaryFieldSupported = {}, dictionaryDataComposites = {}, checkCompositeDataAllowed) {
        if (!pageInstance || !this._storageKey) {
            throw new xtrem_shared_1.SystemError('Invalid arguments');
        }
        try {
            this._dispatchInProgress = false;
            this._initializationInProgress = true;
            this._checkCompositeDataAllowed = checkCompositeDataAllowed;
            // ui.console.warn(`Field(s) supported :\n${JSON.stringify(dictionaryFieldSupported)}`);
            // ui.console.warn(`Composite data :\n${JSON.stringify(dictionaryDataComposites)}`);
            // Process composite data before to continue
            if (typesLib.gt(Object.keys(dictionaryDataComposites ?? {}).length, 0)) {
                this.clearCompositeData();
                // Erase storage
                this._removeCompositeDataStorage(pageInstance);
                this._dictionaryDataComposites = dictionaryDataComposites ?? {};
            }
            else {
                this.loadCompositeData(pageInstance);
            }
            if (typesLib.gt(Object.keys(dictionaryFieldSupported ?? {}).length, 0)) {
                await this.setScreenFieldSupported(pageInstance, dictionaryFieldSupported ?? {});
                // mapped data has not deleted after dispatching
                if (this.isExistsDataComposite && this.isExistsFieldsMapped) {
                    if (this._beginDispatching()) {
                        try {
                            // use existing first segment field
                            if (await this._dispatchDataToFields(pageInstance, this._screenFieldsMapped[0].mainField, false).catch(() => false)) {
                                // Done, data has been dispatched
                            }
                        }
                        catch (error) {
                            ui.console.error(`Error dispatching initial data :\n${error}`);
                        }
                        finally {
                            this._endingDispatching();
                        }
                    }
                }
            }
        }
        catch (error) {
            ui.console.error(`Unexpected initialization error :\n${error}`);
        }
        finally {
            this._initializationInProgress = false;
        }
        return true;
    }
    /**
     * Return if abort dispatch session is requested
     */
    get isAbortDispatchInProgress() {
        return this._abortDispatch;
    }
    /**
     * Check if raw data is a composite data
     * TODO: MUST be override by child class
     * @param _rawData
     * @returns
     */
    // eslint-disable-next-line class-methods-use-this
    isCompositeData(_rawData) {
        throw new Error('Method not implemented.');
    }
    /**
     * Return if dispatch session is in progress
     */
    get isDispatchInProgress() {
        return this._dispatchInProgress;
    }
    /**
     *
     * @returns true when some composite data have been stored.
     */
    get isExistsDataComposite() {
        return typesLib.gt(Object.keys(this._dictionaryDataComposites).length, 0);
    }
    /**
     * Check if exists some fields mapped into service,
     * ready to dispatch
     */
    get isExistsFieldsMapped() {
        return typesLib.gt(this._screenFieldsMapped.length, 0);
    }
    /**
     * Check if exists some fields loaded into service
     */
    get isExistsFieldsSupported() {
        return typesLib.gt(Object.keys(this._screenFieldsSupported).length, 0);
    }
    /**
     * Return if initialization is in progress
     */
    get isInitializationInProgress() {
        return this._initializationInProgress;
    }
    /**
     * Load composite data from storage and then delete it there.
     * @returns true when some data has been loaded
     */
    loadCompositeData(pageInstance) {
        this.clearCompositeData();
        try {
            const dictionaryDataComposites = JSON.parse((pageInstance.$.storage.get(this._storageKey) ?? '{}'));
            // ui.console.warn(`Composite data loaded :\n${JSON.stringify(dictionaryDataComposites)}`);
            // Erase storage
            this._removeCompositeDataStorage(pageInstance);
            if (typesLib.gt(Object.keys(dictionaryDataComposites).length, 0)) {
                this._dictionaryDataComposites = dictionaryDataComposites;
                // ui.console.warn(`Assigned :\n${JSON.stringify(this._dictionaryDataComposites)}`);
                return true;
            }
        }
        catch (error) {
            this.clearCompositeData();
            this._removeCompositeDataStorage(pageInstance);
            ui.console.error(`Error loading composite data from storage :\n${error}`);
        }
        // ui.console.error(`No composite data`);
        return this.isExistsDataComposite;
    }
    /**
     * Save composite data, zero elements allowed
     * @param pageInstance current page
     * @returns true when operation has performed
     */
    saveCompositeData(pageInstance) {
        if (pageInstance && this._storageKey && this.isExistsDataComposite) {
            const dictionaryDataComposites = this._dictionaryDataComposites;
            pageInstance.$.storage.set(this._storageKey, JSON.stringify(dictionaryDataComposites));
            return true;
        }
        if (pageInstance) {
            this._removeCompositeDataStorage(pageInstance);
        }
        return false;
    }
    /**
     * Called when field update is in progress
     * @param pageInstance current page
     * @param currentField reference of field to updating
     * @param rawData received raw data
     * @param appendCompositeData true when is necessary to appending current date to previous set
     * @return true when is not necessary to continue and affect a value : composite value has been processed
     */
    async scan(pageInstance, currentField, rawData, appendCompositeData = false) {
        if (this._beginDispatching()) {
            try {
                let result = true;
                // when is not a composite data, is not necessary to process
                if (!this.isCompositeData(rawData) ||
                    !(await this._decodeCompositeData(rawData, appendCompositeData, currentField.id))) {
                    this._endingDispatching();
                    return false;
                }
                // When composite data have skipped, nothing to do, report no errors
                if (this.isExistsDataComposite) {
                    let _postActionToDo;
                    const _soundOriginalValue = await BarcodeManagementService._setSoundField(pageInstance, currentField, true);
                    try {
                        result = await this._dispatchDataToFields(pageInstance, currentField).catch(() => false);
                        // Special action to do after exiting a function
                        if (result && this.isAbortDispatchInProgress && this._postActionAbortDispatch) {
                            _postActionToDo = this._postActionAbortDispatch;
                        }
                        if (result) {
                            await pageInstance.$.sound.success();
                        }
                        else {
                            await pageInstance.$.sound.error();
                        }
                    }
                    finally {
                        setTimeout(() => {
                            // Execute async operations with proper error handling
                            (async () => {
                                try {
                                    await BarcodeManagementService._setSoundField(pageInstance, currentField, _soundOriginalValue);
                                    if (_postActionToDo)
                                        await _postActionToDo();
                                }
                                catch (error) {
                                    ui.console.error(`Dispatching post done action error :\n${error}`);
                                }
                            })().catch(error => {
                                ui.console.error(`Unexpected error in setTimeout callback: ${error}`);
                            });
                        }, 100);
                    }
                }
                else {
                    await BarcodeManagementService._clearField(pageInstance, currentField);
                }
                return result;
            }
            catch (error) {
                ui.console.error(`Dispatching error :${error}`);
            }
            finally {
                this._endingDispatching();
            }
        }
        ui.console.error('Dispatching not available');
        return false;
    }
    /**
     * Update screen fields supported.
     * field mapping was done after
     * @param pageInstance current page
     * @param dictionaryFieldSupported dictionary of page fields submitted
     * @returns true when done, false when no any field to process with data
     */
    // eslint-disable-next-line require-await
    async setScreenFieldSupported(pageInstance, dictionaryFieldSupported) {
        // Get only fields matching with active parameters
        const lastInitializationInProgress = this.isInitializationInProgress;
        try {
            this._initializationInProgress = true;
            this._screenFieldsSupported =
                this._existingFields(pageInstance, dictionaryFieldSupported).map((field, index) => {
                    const segmentsSupported = [];
                    // Create segments allowed
                    field.segments.forEach(segment => {
                        const { currentField } = segment;
                        segmentsSupported.push({
                            currentField,
                        });
                    });
                    return {
                        DataTitle: field.DataTitle,
                        sequence: typesLib.add(index, 1),
                        segments: segmentsSupported,
                    };
                }) ?? [];
            // ui.console.debug(`Fields supported into page : ${JSON.stringify(this._screenFieldsSupported)}`);
            // Map fields with existing composite data
            return this._mappingFields();
        }
        catch (error) {
            ui.console.error(`Error in setScreenFieldSupported :\n${error}`);
            return false;
        }
        finally {
            this._initializationInProgress = lastInitializationInProgress;
        }
    }
    /**
     *  Return type of service barcode
     *  @returns type of service barcode
     */
    get typeOfServiceBarcode() {
        return this._barcodeManagementConfiguration?.typeOfServiceBarcode ?? 'gs1';
    }
    /**
     * This function start dispatching
     * @returns true when a new session has started, false when already in progress
     */
    /** @internal */
    _beginDispatching() {
        const oldDispatchInProgress = this._dispatchInProgress;
        this._dispatchInProgress = true;
        this._abortDispatch = false;
        this._postActionAbortDispatch = undefined;
        return !oldDispatchInProgress;
    }
    /**
     * Clear current UI field with null value and update page.
     * @param pageInstance current page
     * @param currentField current field to clear
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     */
    /** @internal */
    static async _clearField(pageInstance, currentField, setFocus = true) {
        currentField.value = null;
        // Request all changes to page (optional field and unit) before any action
        await pageInstance.$.commitValueAndPropertyChanges();
        if (setFocus) {
            currentField.focus();
        }
        // Force validation
        await currentField.validate().catch(() => false);
    }
    /**
     * @returns cloned dictionary data composite (must be empty)
     */
    /** @internal */
    _cloneDictionaryDataGeneric() {
        return { ...this._dictionaryDataComposites };
    }
    /**
     * Decode raw data to one or more data fields
     * @param rawData raw composite data before decoding
     * @param appendCompositeData may apply only when valid received data must be replace or append
     * @param scannedFieldId optional field id scanned
     * @returns true when data has valid and operation fully performed
     */
    /** @internal */
    async _decodeCompositeData(rawData, appendCompositeData = false, scannedFieldId) {
        const extractedComposite = {};
        ui.console.debug(`initial data raw : ${JSON.stringify(rawData)}, size : ${rawData.length}`);
        let parsedBarcode;
        try {
            parsedBarcode = this._parseBarCode.parse(rawData);
            ui.console.debug(`RawData :\n${rawData}\nParsed code(s) :\n${JSON.stringify(parsedBarcode)}`);
        }
        catch (error) {
            ui.console.debug(`ERROR in parse :\n${error}`);
            return false;
        }
        /**
         * When the block does not include a prefix or when it is not recognized by the parser,
         * the GS1 composite code block requires in this case more than one AI code to be considered,
         * as a single value could be incorrectly considered composite.
         */
        if (!parsedBarcode.parsedCodeItems.length ||
            (typesLib.lt(parsedBarcode.parsedCodeItems.length, 2) && typesLib.strictEq(parsedBarcode.codeName, ''))) {
            return false;
        }
        // Store data into composite data storage
        parsedBarcode.parsedCodeItems.forEach(parsedCodeItem => {
            // Add a new data entry
            extractedComposite[parsedCodeItem.elementDataTitle] = {
                dataKey: parsedCodeItem.elementDataTitle,
                data: parsedCodeItem.data,
                unit: parsedCodeItem.unit,
            };
        });
        // ui.console.warn(`composite data existing : ${JSON.stringify(this._dictionaryDataComposites)}`);
        // ui.console.warn(`composite data loaded : ${JSON.stringify(extractedComposite)}`);
        // Dictionary update
        if (!appendCompositeData) {
            this.clearCompositeData();
            this._dictionaryDataComposites = extractedComposite;
        }
        else {
            Object.assign(this._dictionaryDataComposites, extractedComposite);
        }
        // ui.console.debug(`composite data updated : ${JSON.stringify(this._dictionaryDataComposites)}`);
        // before to map, check if allowed
        if (!(await this._isDataCompositeAllowed(scannedFieldId))) {
            this.clearCompositeData();
            return true;
        }
        // mapping composite data to dataField without checking error condition
        this._mappingFields();
        // Sometimes no data is stored due to inactive code, this is not an error
        return true;
    }
    /**
     * Terminate current dispatch session
     * @returns true when session has been terminated, false when non session exist
     */
    /** @internal */
    _endingDispatching() {
        const oldDispatchInProgress = this._dispatchInProgress;
        this._dispatchInProgress = false;
        this._abortDispatch = false;
        this._postActionAbortDispatch = undefined;
        return oldDispatchInProgress;
    }
    /**
     * Process client callback when exists to check data
     * @param scannedFieldId optional field id scanned
     * @returns Checks if the composite dataset can be processed
     */
    /** @internal */
    async _isDataCompositeAllowed(scannedFieldId) {
        // If any error occurs, the composite data block will be processed
        try {
            if (this._checkCompositeDataAllowed &&
                this.isExistsDataComposite &&
                !(await this._checkCompositeDataAllowed(this._cloneDictionaryDataGeneric(), scannedFieldId))) {
                return false;
            }
        }
        catch (error) {
            ui.console.error(`Error when evaluating data composite condition is allowed :\n${error}`);
        }
        return true;
    }
    /**
     *
     * @param field check current field is available
     * @returns true if not available
     */
    /** @internal */
    static _isFieldNotAvailable(field) {
        return (field?.isHidden || field?.isDisabled || field?.isReadOnly) ?? true;
    }
    /**
     * returns if the field support sound management
     * @param originalField
     * @returns
     */
    /** @internal */
    static _getIsSupportedSoundField(currentField) {
        return (currentField instanceof ui.fields.Reference ||
            currentField instanceof ui.fields.FilterSelect ||
            currentField instanceof ui.fields.Select ||
            currentField instanceof ui.fields.DropdownList);
    }
    /**
     * This function map current fields with set of data
     * @returns true when exists some fields with data mapped
     */
    /** @internal */
    _mappingFields() {
        // ui.console.debug(`Starting mapping, composite data : ${JSON.stringify(this._dictionaryDataComposites)}`);
        // ui.console.debug(`Screen field(s) supported : ${JSON.stringify(this._screenFieldsSupported)}`);
        // Extract active field
        this._screenFieldsMapped = [];
        const mappingFields = [];
        // Hidden fields are also taken into account because at this stage, we do not know the status changes
        // that will occur during the processing of the various fields.
        this._screenFieldsSupported
            .sort((field1, field2) => typesLib.sub(field1.sequence, field2.sequence))
            .forEach((item, idx) => {
            const dataFieldMapped = {
                DataTitle: item.DataTitle,
                sequence: typesLib.add(idx, 1),
                mainField: item.segments[0].currentField,
                data: this._dictionaryDataComposites[item.DataTitle]?.data,
            };
            // Add optional part
            if (typesLib.gt(item.segments.length, 1) && item.segments[1]?.currentField) {
                dataFieldMapped.unitField = item.segments[1].currentField;
                dataFieldMapped.unit = this._dictionaryDataComposites[item.DataTitle]?.unit;
            }
            mappingFields.push(dataFieldMapped);
        });
        this._screenFieldsMapped = mappingFields;
        // ui.console.debug(`Field(s) Mapped : ${JSON.stringify(this._screenFieldsMapped)}`);
        return typesLib.gt(this._screenFieldsMapped.length, 0);
    }
    /**
     * When possible, changes the sound mode activation state and returns the old value
     * @param currentField current field or undefined
     * @param disableSound true for disable sound for this field
     * @returns old value
     */
    /** @internal */
    static async _setSoundField(pageInstance, currentField, disableSound, refreshField = true) {
        const originalIsSoundDisabled = currentField?.isSoundDisabled ?? false;
        if (currentField && typesLib.strictNe(disableSound, originalIsSoundDisabled) &&
            BarcodeManagementService._getIsSupportedSoundField(currentField)) {
            currentField.isSoundDisabled = disableSound;
            if (refreshField) {
                await pageInstance.$.commitValueAndPropertyChanges();
            }
        }
        return originalIsSoundDisabled;
    }
    /**
     * Remove fields definition
     * @param removeCompositeData  : optionally remove composite data too
     */
    clearScreenFields(removeCompositeData = false) {
        this._screenFieldsSupported = [];
        this._screenFieldsMapped = [];
        if (removeCompositeData) {
            this.clearCompositeData();
        }
    }
    /**
     *  Analyze all fields existing
     * @param pageInstance current page
     * @param dictionaryFieldToCheck
     * @returns
     */
    /** @internal */
    _existingFields(pageInstance, dictionaryFieldToCheck) {
        const $items = pageInstance?._pageMetadata.layout.$items ?? [];
        const preparedFields = this._prepareSegment(dictionaryFieldToCheck);
        const selectedFields = [];
        const dictionarySegment = this._dictionarySegment;
        // ui.console.warn(`Dictionary prepared :\n${JSON.stringify(dictionarySegment)}`);
        // ui.console.warn(`Field prepared :\n${JSON.stringify(preparedFields)}`);
        // For now, the check does not test if the current field is available or not
        if (typesLib.gt($items.length, 0) && typesLib.gt(preparedFields.length, 0)) {
            // Start analysis
            this._existFields(pageInstance, $items, dictionarySegment);
            // retain entries only with full segments
            preparedFields.forEach(fields => {
                if (fields.segments.every(thisSegment => typesLib.strictEq(dictionarySegment[thisSegment.currentField.id]?.currentField.id, thisSegment.currentField.id))) {
                    fields.sequence = dictionarySegment[fields.segments[0].currentField.id].sequence;
                    selectedFields.push(fields);
                }
                else {
                    // Remove partial entries
                    fields.segments.forEach(segment => {
                        delete dictionarySegment[segment.currentField.id];
                    });
                }
            });
        }
        // Update internal dictionary now
        this._dictionarySegment = dictionarySegment;
        // ui.console.debug(`Dictionary updated :\n${JSON.stringify(this._dictionarySegment)}`);
        // ui.console.debug(`Prepared fields :\n${JSON.stringify(preparedFields)}`);
        // ui.console.debug(`Selected fields :\n${JSON.stringify(selectedFields)}`);
        // Sort by sequence
        return selectedFields.sort((field1, field2) => typesLib.sub(field1.sequence, field2.sequence));
    }
    /**
     * Analyze list of fields submitted (recursive)
     * @param pageInstance current page
     * @param $items
     * @param segments current segments list for field
     * @param action perform action
     * @returns
     */
    /** @internal */
    _existFields(pageInstance, $items, segments, depth = 0) {
        // Index from array
        let _index = depth;
        return $items.some(item => {
            _index = typesLib.add(_index, 1);
            const { $bind, $layout } = item;
            if ($bind) {
                const pageField = pageInstance[$bind];
                const field = pageField ? segments[pageField.id] : undefined;
                if (field) {
                    // ui.console.debug(`Bind field : ${JSON.stringify($bind)}`);
                    // Update sequence number, and exit when all fields have been found
                    field.sequence = _index;
                    if (!Object.entries(segments).some(([_key, value]) => typesLib.lte(value.sequence, 0))) {
                        return true;
                    }
                    return false;
                }
            }
            // ui.console.debug(`Skipping bind field : ${JSON.stringify($bind)}`);
            const $items2 = $layout?.$items ?? [];
            // recursively search for 'bind' components inside the layout's containers like section & blocks
            if (typesLib.gt($items2.length, 0) && this._existFields(pageInstance, $items2, segments, _index)) {
                return true;
            }
            _index = typesLib.add(_index, $items2.length);
            return false;
        });
    }
    /**
     * assign date to page fields when existing and available
     * @param pageInstance current page
     * @param originalField original field (reserved)
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     * @returns false when dispatching failure occurs
     */
    /** @internal */
    async _dispatchDataToFields(pageInstance, originalField, setFocus = false) {
        const $items = pageInstance?._pageMetadata.layout.$items ?? [];
        let isFailure = false;
        let isOneFieldProcessed = false;
        this._scannedFieldId = originalField?.id;
        // ui.console.debug('Starting dispatching all fields');
        if (typesLib.gt($items?.length, 0) && this.isExistsDataComposite) {
            // Process fields sequentially using reduce for async sequential processing
            const fieldsWithData = this._screenFieldsMapped.filter(field => field?.data);
            await fieldsWithData.reduce(async (previousPromise, field) => {
                await previousPromise; // Wait for previous field to complete
                if (!isFailure) {
                    // Only continue if no failure yet
                    const success = await this._processField(pageInstance, field, setFocus).catch(() => false);
                    if (!success) {
                        isFailure = true;
                        // ui.console.error(`Dispatching error, composite data has cleared`);
                        this.clearCompositeData();
                    }
                    else {
                        isOneFieldProcessed = true;
                    }
                }
            }, Promise.resolve());
        }
        this._scannedFieldId = undefined;
        // If no composite data could be assigned, it is an error
        if (!isFailure && !isOneFieldProcessed) {
            isFailure = true;
            this.clearCompositeData();
            await BarcodeManagementService._clearField(pageInstance, originalField, setFocus);
            ui.console.error(`Composite data was removed because none could be assigned, and field ${originalField?.id} was reset`);
        }
        // ui.console.debug(`Ending dispatching all fields : ${!isFailure}`);
        return !isFailure;
    }
    /**
     * Convert a user field submitted by full segment reference before evaluation.
     * @param dictionaryFieldsToPrepare fields to check to put in dictionary
     * @returns Screen fields supported for internal use
     */
    /** @internal */
    _prepareSegment(dictionaryFieldsToPrepare) {
        // Reset internal dictionary
        this._dictionarySegment = {};
        const screenFieldsSupported = [];
        Object.entries(dictionaryFieldsToPrepare).forEach(([key, field]) => {
            if (field) {
                const screenFieldSupported = {
                    DataTitle: key,
                    sequence: -1,
                    segments: [],
                };
                const mainField = field.mainField;
                // Create directory entry now
                screenFieldSupported.segments.push((this._dictionarySegment[mainField.id] = {
                    currentField: mainField,
                    sequence: -2,
                }));
                // Add unit too
                const unitField = field.unitField;
                if (unitField) {
                    screenFieldSupported.segments.push((this._dictionarySegment[unitField.id] = {
                        currentField: unitField,
                        sequence: -3,
                    }));
                }
                screenFieldsSupported.push(screenFieldSupported);
            }
        });
        return screenFieldsSupported;
    }
    /**
     *
     * @param pageInstance current page
     * @param field current field already evaluated
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     * @returns true when item has found, but updated only when available
     */
    /** @internal */
    async _processField(pageInstance, field, setFocus = false) {
        const fieldsToRevalidate = [];
        const mainField = field?.mainField;
        const mainFieldId = mainField?.id ?? '';
        const unitField = field?.unitField;
        let validation = true;
        let mainSoundOriginalValue = false;
        let unitSoundOriginalValue;
        try {
            // process all subFields (segments) requested (main and optional unit)
            // ui.console.debug(`_ProcessField (mapped fields) :\n${JSON.stringify(field)}`);
            // current field not currently available : don't process this field
            if (!mainField || BarcodeManagementService._isFieldNotAvailable(mainField)) {
                // ui.console.error(`Field ${mainFieldId} has unavailable`);
                return true;
            }
            if (field?.data) {
                if (unitField && field?.unit && BarcodeManagementService._isFieldNotAvailable(unitField)) {
                    // ui.console.error(`Unit field ${unitFieldId} has unavailable`);
                    return true;
                }
                mainSoundOriginalValue = await BarcodeManagementService._setSoundField(pageInstance, mainField, true);
                // If not possible to change a value, composite block is invalid
                switch (await BarcodeManagementService._setFieldValue(pageInstance, mainField, field?.data, setFocus)) {
                    case SetFieldResult.noData:
                        // ui.console.warn(`Field ${mainFieldId} has no data available`);
                        return false;
                    case SetFieldResult.error:
                        // The received value could not be processed.
                        validation = false;
                        break;
                    case SetFieldResult.unchanged:
                    // force validation too
                    // eslint-disable-next-line no-fallthrough
                    case SetFieldResult.updated:
                        fieldsToRevalidate.push(mainField);
                        break;
                    default:
                        // Unexpected result, treat as error
                        validation = false;
                        break;
                }
                // Process optional unit available with data
                if (validation &&
                    unitField &&
                    field?.unit &&
                    !BarcodeManagementService._isFieldNotAvailable(unitField)) {
                    unitSoundOriginalValue = await BarcodeManagementService._setSoundField(pageInstance, unitField, true);
                    switch (await BarcodeManagementService._setFieldValue(pageInstance, unitField, field.unit)) {
                        case SetFieldResult.noData:
                            // ui.console.warn(`Unit field ${unitFieldId} has no data available`);
                            return false;
                        case SetFieldResult.error:
                            // The received value could not be processed.
                            validation = false;
                            break;
                        case SetFieldResult.unchanged:
                            // force validation too
                            fieldsToRevalidate.push(unitField);
                            break;
                        case SetFieldResult.updated:
                            fieldsToRevalidate.push(unitField);
                            break;
                        default:
                            // Unexpected result, treat as error
                            validation = false;
                            break;
                    }
                }
                // Request all changes to page (optional field and unit) before any action
                await pageInstance.$.commitValueAndPropertyChanges();
                // We only process with the case of modified fields
                if (fieldsToRevalidate.length) {
                    // Now process change / validation field segments
                    // Process each field sequentially, checking for abort after each
                    await fieldsToRevalidate.reduce(async (promise, fieldToRevalidate) => {
                        await promise; // Wait for previous to complete
                        if (this.isAbortDispatchInProgress) {
                            return; // Stop processing
                        }
                        const fieldId = fieldToRevalidate.id;
                        try {
                            // This is to protect against the case where the executeOnChange function does not exist on the field type
                            // if new ones are added later.
                            if (typesLib.strictNe(typeof fieldToRevalidate.executeOnChange, 'function')) {
                                throw new xtrem_shared_1.SystemError(`Field ${fieldId} does not implement executeOnChange.`);
                            }
                            await fieldToRevalidate.executeOnChange(false);
                        }
                        catch (error) {
                            const _info = JSON.stringify(error);
                            const _error = typesLib.strictEq(_info, '{}') ? error.message : _info;
                            throw new xtrem_shared_1.SystemError(`executeOnChange failure on field ${fieldId} :\n${_error}`, error);
                        }
                        if (this.isAbortDispatchInProgress) {
                            // ui.console.warn(`Aborting dispatching in progress for field ${fieldId}`);
                        }
                    }, Promise.resolve());
                    await pageInstance.$.commitValueAndPropertyChanges();
                    // CAUTION: when aborting is in progress, context may be invalid :
                    // This behavior is taken into account by the possibility of abandoning dispatching,
                    // and possibly executing a postDone action at the end. If possible, this action should
                    // avoid being asynchronous for this reason.
                    if (!this.isAbortDispatchInProgress) {
                        // ui.console.debug(`Starting Validation`);
                        // Process validation sequentially, stopping at first failure
                        await fieldsToRevalidate.reduce(async (promise, fieldToValidate) => {
                            await promise; // Wait for previous validation
                            if (!validation) {
                                return; // Stop if already failed
                            }
                            const validationResult = await fieldToValidate
                                .validate()
                                .then(data => typesLib.strictEq(data, undefined))
                                .catch(() => false);
                            validation &&= validationResult;
                            if (!validation) {
                                ui.console.warn(`Validation failure on field : ${fieldToValidate.id}`);
                            }
                        }, Promise.resolve());
                    }
                }
            }
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : String(error);
            ui.console.error(`Error dispatching ${mainFieldId} :\n${errorMessage}`);
            return false;
        }
        finally {
            let isChanged = typesLib.strictNe((await BarcodeManagementService._setSoundField(pageInstance, mainField, mainSoundOriginalValue, false)), mainSoundOriginalValue);
            if (typesLib.strictNe(unitSoundOriginalValue, undefined)) {
                isChanged ||= typesLib.strictNe((await BarcodeManagementService._setSoundField(pageInstance, unitField, unitSoundOriginalValue, false)), unitSoundOriginalValue);
            }
            if (isChanged) {
                await pageInstance.$.commitValueAndPropertyChanges();
            }
        }
        return validation;
    }
    /**
     * Remove data from the storage
     * @param pageInstance current page
     */
    /** @internal */
    _removeCompositeDataStorage(pageInstance, storageKey) {
        const isDataStorage = !!pageInstance && (!!this._storageKey || !!storageKey);
        if (isDataStorage) {
            pageInstance.$.storage.remove(storageKey ?? this._storageKey);
        }
        return isDataStorage;
    }
    /**
     * Assign value to field
     * @param field current field
     * @param value value to assign in the field
     * @param setFocus optional value must be true when focus must be performed in reInitializing
     * @return SetFieldResult.nodata if field has reinitialized, SetFieldResult.noChange when no change has performed
     */
    /** @internal */
    static async _setFieldValue(pageInstance, field, value, setFocus = true) {
        const originalValue = field.value;
        let newValue;
        if (field instanceof ui.fields.Reference) {
            const result = await field.fetchSuggestions(value);
            // We only update if there is a result otherwise the value will remain unchanged
            if (!result.length) {
                await BarcodeManagementService._clearField(pageInstance, field, setFocus);
                return SetFieldResult.noData;
            }
            /* Duplicate not supported, keep first result */
            [newValue] = result;
        }
        else if (field instanceof ui.fields.FilterSelect ||
            field instanceof ui.fields.Select ||
            field instanceof ui.fields.DropdownList) {
            newValue = value;
        }
        else if (field instanceof ui.fields.Text) {
            newValue = String(value);
        }
        else if (field instanceof ui.fields.Numeric) {
            newValue = Number(value);
        }
        else if (field instanceof ui.fields.Date) {
            newValue = value instanceof xtrem_date_time_1.DateValue ? value.toString() : value;
        }
        else {
            newValue = value;
        }
        field.value = newValue;
        const isChanged = typesLib.strictNe(originalValue, newValue);
        if (isChanged && !field.isDirty) {
            field.isDirty = true;
        }
        return isChanged ? SetFieldResult.updated : SetFieldResult.unchanged;
    }
}
exports.BarcodeManagementService = BarcodeManagementService;
//# sourceMappingURL=barcode-management-service.js.map