"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BarcodeConfigurator = void 0;
const typesLib = require("@sage/xtrem-decimal");
const ui = require("@sage/xtrem-ui");
const barcode_parser_configuration_1 = require("../shared-functions/barcode-parser-configuration");
const screen_management_1 = require("./screen-management");
// FIXME: TODO: Start block - Add the missing type before the implementation
var CompositeCodeType;
(function (CompositeCodeType) {
    CompositeCodeType["delimited"] = "delimited";
    CompositeCodeType["fixedLength"] = "fixedLength";
})(CompositeCodeType || (CompositeCodeType = {}));
class BarcodeConfigurator {
    constructor() {
        this._barcodeConfiguration = undefined;
    }
    /**
     *  Initialize the barcode configuration
     *
     * Configurations are returned only if they are all defined:
     * - Barcode configuration must be read and have lines.
     * - Parser configuration must be able to be generated.
     * - Field configuration is only generated if all provided fields were found on the screen.
     * @param pageInstance
     * @param codeFlow
     * @returns instance of barcode configuration or undefined
     */
    async initConfigurator(pageInstance, codeFlow) {
        const _barcodeConfiguration = await this._getBarcodeConfiguration(pageInstance, codeFlow);
        const _parserConfiguration = _barcodeConfiguration
            ? BarcodeConfigurator._computeParserConfiguration(_barcodeConfiguration)
            : undefined;
        const _barcodeManagementConfiguration = _barcodeConfiguration && _parserConfiguration
            ? this._computeManagementConfiguration(pageInstance, _barcodeConfiguration, _parserConfiguration)
            : undefined;
        if (!_barcodeConfiguration || !_parserConfiguration || !_barcodeManagementConfiguration) {
            this._parserConfiguration = undefined;
            this._barcodeConfiguration = undefined;
            this._barcodeManagementConfiguration = undefined;
            return undefined;
        }
        this._barcodeConfiguration = _barcodeConfiguration;
        this._parserConfiguration = _parserConfiguration;
        this._barcodeManagementConfiguration = _barcodeManagementConfiguration;
        ui.console.debug(`The barcode configuration ${_barcodeConfiguration.compositeCode ?? ''} of type ${_barcodeManagementConfiguration.typeOfServiceBarcode} has been loaded with ${_barcodeConfiguration.lines.length} parameter(s).`);
        return _barcodeConfiguration;
    }
    /**
     *  Compute the parser configuration :
     *  No configuration is returned if no rows could be retrieved.
     * @param _barcodeConfiguration
     *  @returns parser configuration or undefined
     */
    static _computeParserConfiguration(_barcodeConfiguration) {
        const _lines = [];
        const isFixedLength = typesLib.strictEq(_barcodeConfiguration.compositeFormat, barcode_parser_configuration_1.CompositeFormat.fixedLength);
        const _dateSize = (0, barcode_parser_configuration_1.getDateFormatSize)(_barcodeConfiguration.dateFormat);
        const _decimalSeparator = _barcodeConfiguration?.decimalSeparator;
        const _fieldSeparator = _barcodeConfiguration?.fieldSeparator;
        let _recordSize = 0;
        let _numberOfAlias = 0;
        if (!_dateSize) {
            ui.console.error('Date format is not defined');
            return undefined;
        }
        if (!_decimalSeparator) {
            ui.console.error('Decimal separator is not defined');
            return undefined;
        }
        if (!isFixedLength) {
            if (!_fieldSeparator) {
                ui.console.error('Field separator is not defined');
                return undefined;
            }
            if (typesLib.strictEq(_fieldSeparator, _decimalSeparator)) {
                ui.console.error('Field separator is the same as the decimal separator');
                return undefined;
            }
        }
        // Check for date field length validation errors first
        const dateFieldError = _barcodeConfiguration.lines.find(_line => {
            return typesLib.strictEq(_line.fieldDataType, 'date') && isFixedLength && typesLib.gt(_dateSize, _line.length);
        });
        if (dateFieldError) {
            ui.console.error(`Date field is too short for field : ${dateFieldError.fieldName}`);
            return undefined;
        }
        // Process all lines to build parser configuration
        _barcodeConfiguration.lines.forEach(_line => {
            if (isFixedLength) {
                _recordSize = typesLib.add(_recordSize, _line.length);
            }
            if (_line.fieldAlias) {
                _numberOfAlias = typesLib.add(_numberOfAlias, 1);
            }
            _lines.push({
                fieldKey: _line.fieldKey,
                description: _line.fieldName,
                ...(isFixedLength && { startPosition: _line.startPosition }),
                ...(isFixedLength && { length: _line.length }),
                fieldAlias: _line?.fieldAlias,
                fieldDataType: _line.fieldDataType,
            });
        });
        if (!isFixedLength && _lines.length) {
            _recordSize = typesLib.add(_recordSize, typesLib.mul((typesLib.sub(_lines.length, 1)), _fieldSeparator.length));
        }
        return _lines.length
            ? {
                typeOfBarcodeParser: 'composite',
                compositeCode: _barcodeConfiguration.compositeCode,
                localizedDescription: _barcodeConfiguration.localizedDescription,
                isBlankDeletion: _barcodeConfiguration.isBlankDeletion,
                compositeFormat: _barcodeConfiguration.compositeFormat,
                dateFormat: _barcodeConfiguration.dateFormat,
                dateLabel: _barcodeConfiguration.dateLabel,
                ...(!isFixedLength && { fieldSeparator: _fieldSeparator }),
                decimalSeparator: _decimalSeparator,
                recordSize: _recordSize,
                numberOfAlias: _numberOfAlias,
                lines: _lines,
            }
            : undefined;
    }
    /**
     *  Compute the screen management configuration
     * @param _barcodeConfiguration
     */
    _computeManagementConfiguration(pageInstance, _barcodeConfiguration, _parserConfiguration) {
        const _dictionary = {};
        let numberOfAlias = 0;
        // Use find to detect early return conditions
        const hasError = _barcodeConfiguration.lines.find(_line => {
            const _fieldName = _line.fieldName;
            if (_line.fieldKey && _fieldName) {
                const _fieldAlias = _line.fieldAlias;
                const _fieldDataType = _line.fieldDataType;
                // We start with the main field
                const _mainField = pageInstance[_fieldName];
                if (!_mainField?.id) {
                    ui.console.error(`Field ${_fieldName} is not defined in the page`);
                    return false; // Skip this line but continue
                }
                if (!(0, screen_management_1.isSupportedField)(_mainField)) {
                    ui.console.error(`Field ${_fieldName} is not supported by the screen management`);
                    return false; // Skip this line but continue
                }
                const _mainFieldRightType = this._getSupportedFieldDateType(_mainField);
                // Check if the field type is known directly
                if (_mainFieldRightType && typesLib.strictNe(_fieldDataType, _mainFieldRightType)) {
                    ui.console.error(`The type ${_fieldDataType} of the field ${_fieldName} does not match that of the page ${_mainFieldRightType}.`);
                    return true; // Signal error to abort entire function
                }
                _dictionary[_line.fieldKey] = {
                    mainField: _mainField,
                };
                // If an alias is defined, we also check if it is supported
                if (_fieldAlias) {
                    const _aliasField = pageInstance[_fieldAlias];
                    if (!_aliasField?.id) {
                        ui.console.error(`Field alias ${_fieldAlias} of ${_fieldName} is not defined in the page`);
                        return false; // Skip this alias but continue
                    }
                    if (!(0, screen_management_1.isSupportedField)(_aliasField)) {
                        ui.console.error(`Field alias ${_fieldAlias} of ${_fieldName} is not supported by the screen management`);
                        return false; // Skip this alias but continue
                    }
                    const _aliasFieldRightType = this._getSupportedFieldDateType(_aliasField);
                    // Check if the field type is known directly
                    if (_aliasFieldRightType && typesLib.strictNe(_fieldDataType, _aliasFieldRightType)) {
                        ui.console.error(`The type ${_fieldDataType} of the field alias ${_fieldAlias} of ${_fieldName}  does not match that of the page ${_aliasFieldRightType}.`);
                        return true; // Signal error to abort entire function
                    }
                    _dictionary[_fieldAlias] = {
                        mainField: _aliasField,
                    };
                    numberOfAlias = typesLib.add(numberOfAlias, 1);
                }
            }
            return false; // No error, continue processing
        });
        // If we found an error that requires aborting, return undefined
        if (hasError) {
            return undefined;
        }
        const _isSameSize = typesLib.strictEq(Object.keys(_dictionary).length, typesLib.add(_barcodeConfiguration.lines.length, numberOfAlias));
        if (!_isSameSize) {
            ui.console.error('At least one field could not be handled by the screen management, the configuration is rejected.');
        }
        return _isSameSize
            ? {
                typeOfServiceBarcode: _barcodeConfiguration.typeOfServiceBarcode,
                recordSize: _parserConfiguration.recordSize,
                numberOfAlias,
                dictionaryFieldSupported: _dictionary,
                parserConfiguration: _parserConfiguration,
            }
            : undefined;
    }
    /**
     * Get the barcode configuration
     * @returns current instance of barcode configuration or undefined
     */
    get barcodeConfiguration() {
        return this._barcodeConfiguration;
    }
    /**
     * Get the barcode management configuration
     * @returns current instance of barcode management configuration or undefined
     */
    get barcodeManagementConfiguration() {
        return this._barcodeManagementConfiguration;
    }
    /**
     * Get the dictionary field supported
     * @returns current instance of dictionary field supported or undefined
     */
    get dictionaryFieldSupported() {
        return this._barcodeManagementConfiguration?.dictionaryFieldSupported;
    }
    /**
     * Get the parser configuration
     * @returns current instance of parser configuration or undefined
     */
    get parserConfiguration() {
        return this._parserConfiguration;
    }
    /**
     * Get the type of service barcode
     * @returns current instance of type of service barcode or undefined
     */
    get typeOfServiceBarcode() {
        return this._barcodeManagementConfiguration?.typeOfServiceBarcode;
    }
    /**
     *  Get the barcode configuration for the given code flow for fabricator
     * @param pageInstance
     *  @param codeFlow
     *  @returns
     */
    async _getBarcodeConfiguration(pageInstance, codeFlow) {
        const _barcodeSetup = await this._getBarcodeSetup(pageInstance, codeFlow);
        if (_barcodeSetup) {
            const _lines = [];
            const _decimalSeparator = _barcodeSetup.decimalSeparator?.trim() ?? '.';
            // Field name must be alphanumeric only (including underscore).
            const _regExAlphaNumeric = /^_*[A-Za-z]+\w*$/;
            // Use find to detect validation errors that require early return
            const validationError = _barcodeSetup.lines.find(_line => {
                // Remove all spaces in the field name
                const _fieldName = _line.fieldName?.trim().replace(/ /g, '');
                const _aliasName = _line?.fieldAlias?.trim().replace(/ /g, '');
                const _fieldAlias = _aliasName && typesLib.strictNe(_fieldName, _aliasName) ? _aliasName : undefined;
                const _fieldKey = _fieldName;
                if (_fieldKey && _fieldName) {
                    if (!_regExAlphaNumeric.test(_fieldName)) {
                        ui.console.error(`Field name is not valid: ${_fieldName}`);
                        return true; // Signal validation error
                    }
                    if (_fieldAlias && !_regExAlphaNumeric.test(_fieldAlias)) {
                        ui.console.error(`Field key alias for ${_fieldKey} is not valid: ${_fieldAlias}`);
                        return true; // Signal validation error
                    }
                    _lines.push({
                        fieldKey: _fieldKey,
                        fieldName: _fieldName,
                        fieldAlias: _fieldAlias,
                        description: _line.fieldName?.trim(),
                        startPosition: _line.startPosition,
                        length: _line.length,
                        fieldDataType: this._getSupportedDataType(_line.fieldDataType),
                    });
                }
                return false; // No validation error, continue processing
            });
            // If validation error found, return undefined
            if (validationError) {
                return undefined;
            }
            // The service type returned for now is a constant, as it is not (yet) provided by reading the configuration
            return _lines.length
                ? {
                    typeOfServiceBarcode: 'composite',
                    compositeCode: _barcodeSetup.compositeCode?.trim(),
                    localizedDescription: _barcodeSetup.localizedDescription?.trim(),
                    isActive: _barcodeSetup.isActive,
                    isBlankDeletion: _barcodeSetup.isBlankDeletion,
                    compositeFormat: this._getCompositeFormat(_barcodeSetup.compositeCodeType),
                    dateFormat: this._getCompositeDateFormat(_barcodeSetup.dateFormat),
                    dateLabel: _barcodeSetup.dateLabel.trim(),
                    fieldSeparator: _barcodeSetup.fieldSeparator.trim(),
                    decimalSeparator: typesLib.strictNe(_decimalSeparator, '') ? _decimalSeparator : '.',
                    lines: _lines,
                }
                : undefined;
        }
        return undefined;
    }
    /**
     *  Get the composite format for the given composite code type
     *  @param compositeCodeType
     *  @returns composite format type key
     */
    // eslint-disable-next-line class-methods-use-this
    _getCompositeFormat(compositeCodeType) {
        if (typesLib.strictEq(compositeCodeType, CompositeCodeType.delimited)) {
            return barcode_parser_configuration_1.CompositeFormat.delimited;
        }
        if (typesLib.strictEq(compositeCodeType, CompositeCodeType.fixedLength)) {
            return barcode_parser_configuration_1.CompositeFormat.fixedLength;
        }
        return barcode_parser_configuration_1.CompositeFormat.delimited;
    }
    /**
     * Get the current stock site
     *  @param pageInstance Current page instance
     *  @returns the current stock site
     */
    // eslint-disable-next-line class-methods-use-this
    _getCurrentStockSite(pageInstance) {
        return pageInstance.$.storage.get('mobile-selected-stock-site');
    }
    /**
     *  Get the date format for the given date format
     * @param dateFormat
     * @returns date format type key
     */
    // eslint-disable-next-line class-methods-use-this
    _getCompositeDateFormat(dateFormat) {
        switch (dateFormat) {
            case 1:
                return barcode_parser_configuration_1.CompositeDateFormat.ddmmyy;
            case 2:
                return barcode_parser_configuration_1.CompositeDateFormat.ddmmyyyy;
            case 3:
                return barcode_parser_configuration_1.CompositeDateFormat.ddmmyy;
            case 4:
                return barcode_parser_configuration_1.CompositeDateFormat.yyyymmdd;
            case 5:
                return barcode_parser_configuration_1.CompositeDateFormat.yyyy_mm_dd;
            default:
                return barcode_parser_configuration_1.CompositeDateFormat.yyyymmdd;
        }
    }
    /**
     *  Get the data type for the given field data type
     * @param fieldDataType
     * @returns
     */
    // eslint-disable-next-line class-methods-use-this
    _getSupportedDataType(fieldDataType) {
        switch (fieldDataType) {
            case 'alphanumeric':
                return 'string';
            case 'numeric':
                return 'number';
            case 'date':
                return 'DateValue';
            case 'localMenu':
                return 'string';
            default:
                return 'string';
        }
    }
    /**
     *  Get the data type for the given field data type
     * @param fieldDataType
     * @returns data type key or undefined if type is not supported (not possible to check immediately)
     */
    // eslint-disable-next-line class-methods-use-this
    _getSupportedFieldDateType(fieldDataType) {
        if (!(0, screen_management_1.isSupportedField)(fieldDataType) || (0, screen_management_1.isSoundSupportedField)(fieldDataType)) {
            return undefined;
        }
        if (fieldDataType instanceof ui.fields.Text) {
            return 'string';
        }
        if (fieldDataType instanceof ui.fields.Numeric) {
            return 'number';
        }
        if (fieldDataType instanceof ui.fields.Date) {
            return 'DateValue';
        }
        return undefined;
    }
    /**
     *  Get the barcode configuration for the given code flow
     * @param pageInstance
     * @param codeFlow
     * @returns
     */
    // eslint-disable-next-line require-await
    async _getBarcodeSetup(pageInstance, codeFlow) {
        try {
            const _codeSite = this._getCurrentStockSite(pageInstance);
            if (!_codeSite) {
                ui.console.error('Stock site is not defined');
                return undefined;
            }
            if (!codeFlow) {
                ui.console.error('Code flow is not defined');
                return undefined;
            }
            // FIXME: TODO: Start of block - Node to must implemented
            // return extractEdges<any>(
            //    await pageInstance.$.graph
            //        .node('@sage/wh-system/CompositeCodeSetup')
            //        .query(
            //            ui.queryUtils.edgesSelector(
            //                {
            //                    flow: { code: true },
            //                    stockSite: { code: true },
            //                    isActive: true,
            //                    compositeCode: true,
            //                    localizedDescription: true,
            //                    isBlankDeletion: true,
            //                    compositeCodeType: true,
            //                    dateFormat: true,
            //                    dateLabel: true,
            //                    fieldSeparator: true,
            //                    decimalSeparator: true,
            //                    lines: {
            //                        query: {
            //                            edges: {
            //                                node: {
            //                                    lineNumber: true,
            //                                    fieldName: true,
            //                                    fieldAlias: true,
            //                                    startPosition: true,
            //                                    length: true,
            //                                    fieldDataType: true,
            //                                },
            //                            },
            //                            __args: {
            //                                first: 200,
            //                            },
            //                        },
            //                    },
            //                },
            //                {
            //                    first: 2,
            //                    filter: {
            //                        flow: { code: codeFlow },
            //                        stockSite: { code: { _in: [_codeSite, undefined] } },
            //                        isActive: true,
            //                    },
            //                    orderBy: { stockSite: { code: -1 } },
            //                },
            //            ),
            //        )
            //        .execute(),
            // )[0];
            // FIXME: TODO: End of block - Node to must implemented
        }
        catch (error) {
            ui.console.error(`Error getting barcode configuration: ${error instanceof Error ? error.message : String(error)}`);
        }
        return undefined;
    }
}
exports.BarcodeConfigurator = BarcodeConfigurator;
//# sourceMappingURL=barcode-configurator.js.map