import type { Depositor, GraphApi, WhSite } from '@sage/wh-master-data-api';
import type { User } from '@sage/wh-system-api';
import * as ui from '@sage/xtrem-ui';
import {
    SiteDepositor,
    getSelectedSiteDepositor,
    removeSelectedSiteDepositor,
    saveSelectedSiteDepositor,
    setSelectedSiteDepositor,
} from '../client-functions/storage-properties';
import { getDefaultSite, readDepositor, readUser } from '../client-functions/user-extension';

@ui.decorators.sticker<MobileSiteDepositor>({
    title: 'Site and depositor',
    icon: 'settings',
    category: 'mobile',
    isActive() {
        return true;
    },
    onClose() {
        this.$.removeToasts(); // Remove all toasts
    },
    async onLoad() {
        let selectedSiteDepositor = getSelectedSiteDepositor(this);
        const user = await readUser(this, this.$.userCode?.toUpperCase() ?? '');
        if (user) {
            const isUserAdmin = this._isCurrentUserAdmin();
            this.selectionUser.value = { code: user.code };
            this.selectionUser.isReadOnly = !isUserAdmin;
            this.selectionUser.isDisabled = !isUserAdmin;
            this.selectionUser.isHidden = !isUserAdmin;
            this.selectionUser.isMandatory = isUserAdmin;

            // We first search for the user's current site
            const currentSelectedSite = user.currentSiteCode;
            this._isAllowedSiteChange = isUserAdmin || user.isAllowedSiteChange;
            this._isAllowedDepositorChange = isUserAdmin || user.isAllowedDepositorChange;

            // User does not have a site, ignore temporary rules to allow site and depositor change
            if (!currentSelectedSite) {
                selectedSiteDepositor = undefined;
                removeSelectedSiteDepositor(this);
            }

            // If there is no site in the session storage, retrieve it from the getDefaultSite service
            if (!selectedSiteDepositor || !selectedSiteDepositor?.depositor) {
                let sessionSelectedSite: string | undefined;
                let sessionSelectedDepositor: string | undefined;

                // We first search for the user's current site for the default site
                if (!currentSelectedSite) {
                    if (!isUserAdmin || this._isAllowedDepositorChange) {
                        sessionSelectedSite = user.defaultSiteCode;
                    }
                } else {
                    // Then the user's depositor
                    sessionSelectedSite = currentSelectedSite;
                    sessionSelectedDepositor = user.currentDepositorCode;
                }

                if (!isUserAdmin && (!sessionSelectedSite || !sessionSelectedDepositor)) {
                    selectedSiteDepositor = undefined;
                } else {
                    selectedSiteDepositor = <SiteDepositor>{
                        site: sessionSelectedSite,
                        ...(sessionSelectedDepositor && { depositor: sessionSelectedDepositor }),
                    };
                }
            } else if (selectedSiteDepositor?.depositor) {
                // If the depositor is not valid, reset it
                if (
                    this._isAllowedDepositorChange &&
                    !(await readDepositor(this, selectedSiteDepositor.site, selectedSiteDepositor.depositor))?.code
                ) {
                    if (isUserAdmin) {
                        delete selectedSiteDepositor.depositor;
                    } else {
                        selectedSiteDepositor = undefined;
                    }
                }
            }
        }

        // If the user is not found, lock the screen and reset the site and depositor
        if (!user || !selectedSiteDepositor) {
            selectedSiteDepositor = undefined;
            this._isAllowedSiteChange = false;
            this._isAllowedDepositorChange = false;
        }

        // Update the site, depositor selection fields, and storage.
        this._updateSelectionFieldsSiteDepositorAndStorage(selectedSiteDepositor);

        // Update the save button
        this._updateSelectionDepositorAndSaveButton();
    },
    businessActions() {
        return [this.saveButton];
    },
})
export class MobileSiteDepositor extends ui.Sticker<GraphApi> {
    @ui.decorators.section<MobileSiteDepositor>({
        title: 'Site / depositor',
    })
    section!: ui.containers.Section;

    @ui.decorators.block<MobileSiteDepositor>({
        parent() {
            return this.section;
        },
    })
    block!: ui.containers.Block;

    /**
     * User selection field empty is current user
     */
    @ui.decorators.referenceField<MobileSiteDepositor, User>({
        node: '@sage/wh-system/User',
        valueField: 'code',
        columns: [
            ui.nestedFields.text({ bind: 'code', title: 'Site', canFilter: true }),
            ui.nestedFields.text({ bind: 'name', title: 'Name', canFilter: true }),
        ],
        isFullWidth: true,
        isAutoSelectEnabled: true,
        isHidden: this,
        isDisabled: true,
        isReadOnly: true,
        isMandatory: false,
        title: 'User',
        helperTextField: 'name',
        parent() {
            return this.block;
        },
        async onChange() {
            if (this.selectionUser.value?.code) {
                const user = await readUser(this, this.selectionUser.value.code);
                if (user) {
                    const _userCode = user.code.toUpperCase();
                    // We first search for the user's current site
                    const currentSelectedSite = user.currentSiteCode;
                    const sessionSelectedSite = currentSelectedSite ?? (await getDefaultSite(user));
                    const sessionSelectedDepositor = currentSelectedSite ? user.currentDepositorCode : undefined;
                    const selectedSiteDepositor = sessionSelectedSite
                        ? <SiteDepositor>{
                              site: sessionSelectedSite,
                              ...(sessionSelectedDepositor && { depositor: sessionSelectedDepositor }),
                          }
                        : undefined;

                    // Update the site, depositor selection fields, storage except menu.
                    this._updateSelectionFieldsSiteDepositorAndStorage(
                        selectedSiteDepositor,
                        this._isUserAdmin(_userCode),
                    );

                    // Update the save button
                    this._updateSelectionDepositorAndSaveButton();
                }
            }
        },
    })
    selectionUser!: ui.fields.Reference<User>;

    @ui.decorators.referenceField<MobileSiteDepositor, WhSite>({
        node: '@sage/wh-master-data/WhSite',
        valueField: { site: { code: true } },
        columns: [
            ui.nestedFields.reference({
                title: 'site',
                node: '@sage/wh-system/Site',
                bind: 'site',
                valueField: 'code',
                canFilter: true,
            }),
            ui.nestedFields.reference({
                title: 'name',
                node: '@sage/wh-system/Site',
                bind: 'site',
                valueField: 'name',
                canFilter: true,
            }),
        ],
        isFullWidth: true,
        isAutoSelectEnabled: true,
        title: 'Site',
        helperTextField: { site: { name: true } },
        parent() {
            return this.block;
        },
        async onChange() {
            if (!this._isAllowedDepositorChange && !(await this._checkDepositor())) {
                this._updateSaveButton(true);
            } else {
                this._updateSelectionDepositorAndSaveButton(true);
            }
        },
    })
    selectionSite!: ui.fields.Reference<WhSite>;

    @ui.decorators.referenceField<MobileSiteDepositor, Depositor>({
        node: '@sage/wh-master-data/Depositor',
        // TODO: FIXME: this should be a filter on the site field, but this changing to undefined
        filter() {
            return {
                site: { code: this.selectionSite.value?.site?.code },
            };
        },
        valueField: 'code',
        columns: [
            ui.nestedFields.text({ bind: 'code', title: 'Depositor', canFilter: true }),
            ui.nestedFields.text({ bind: 'shortDescription', title: 'Description', canFilter: true }),
        ],
        isFullWidth: true,
        isAutoSelectEnabled: true,
        isDisabled: true,
        title: 'Depositor',
        helperTextField: 'shortDescription',
        parent() {
            return this.block;
        },
        async onChange() {
            if (this._isAllowedDepositorChange && !(await this._checkDepositor())) {
                this._updateSaveButton(true);
            } else {
                this._updateSaveButton();
            }
        },
    })
    selectionDepositor!: ui.fields.Reference<Depositor>;

    @ui.decorators.pageAction<MobileSiteDepositor>({
        title: 'Set site and depositor',
        buttonType: 'primary',
        isTransient: true,
        isDisabled: true,
        async onClick() {
            const _selectedSiteDepositor = <SiteDepositor>{
                site: this.selectionSite.value?.site?.code,
                ...(this.selectionDepositor.value?.code && { depositor: this.selectionDepositor.value.code }),
            };

            this.$.removeToasts(); // Remove all toasts

            // Attempt to save the selected site and depositor for given user to X3
            if (!(await saveSelectedSiteDepositor(this, this.selectionUser.value?.code, _selectedSiteDepositor))) {
                this.$.showToast(
                    ui.localize(
                        '@sage/wh-master-data/sticker__site_depositor_save_error',
                        'The site / depositor has not been saved.',
                    ),
                    { type: 'error', timeout: 5000 },
                );
            } else {
                // Save the selected site and depositor to the storage
                await this._setMobileSettings(_selectedSiteDepositor);

                // Close the sticker
                this.$.finish();
            }
        },
    })
    saveButton!: ui.PageAction;

    /**
     * Set the selected site and depositor, and update the sticker
     * @param _selectedSiteDepositor
     * @param isUpdateMenu
     */
    private async _setMobileSettings(
        _selectedSiteDepositor: SiteDepositor | undefined,
        isUpdateMenu: boolean = true,
    ): Promise<void> {
        if (_selectedSiteDepositor && _selectedSiteDepositor.site) {
            setSelectedSiteDepositor(this, _selectedSiteDepositor);
            if (isUpdateMenu) {
                this.$.sticker.updateMenuItem(
                    `${_selectedSiteDepositor.site}${_selectedSiteDepositor?.depositor ? ` / ${_selectedSiteDepositor.depositor}` : ''}`,
                );
            }
            this.selectionSite.value = { site: { code: _selectedSiteDepositor.site } };
            this.selectionDepositor.value = _selectedSiteDepositor?.depositor
                ? {
                      code: _selectedSiteDepositor?.depositor,
                      site: { code: _selectedSiteDepositor.site },
                  }
                : null;
        } else {
            this.selectionSite.value = null;
            this.selectionDepositor.value = null;
            removeSelectedSiteDepositor(this);
        }
    }

    /**
     *
     * @returns true if the selected depositor is valid, false otherwise
     */
    private async _checkDepositor(): Promise<boolean> {
        const isDepositorInvalid =
            this._checkSiteDepositorHasValues() &&
            !!(
                await readDepositor(
                    this,
                    this.selectionSite.value?.site?.code ?? '',
                    this.selectionDepositor.value?.code ?? '',
                )
            )?.code;

        // If the depositor for this site is not valid, display an error message
        if (!isDepositorInvalid) {
            this.$.removeToasts(); // Remove all toasts
            this.$.showToast(
                ui.localize(
                    '@sage/wh-master-data/sticker__site_depositor_error',
                    'The site / depositor does not exist.',
                ),
                { type: 'error', timeout: 5000 },
            );
        }

        return isDepositorInvalid;
    }

    /**
     * Check if the user is an admin
     * @param userCode
     * @returns true if the user is an admin, false otherwise
     */
    // eslint-disable-next-line class-methods-use-this
    private readonly _isUserAdmin = (userCode: string | undefined | null) => userCode?.toUpperCase() === 'ADMIN';

    /**
     * Check if the current user is an admin
     * @returns true if the current user is an admin, false otherwise
     */
    private readonly _isCurrentUserAdmin = () => this._isUserAdmin(this.$.userCode);

    /**
     * Update the selectionDepositor and saveButton
     * @param clearDepositor
     * @returns true if the selectionDepositor is enabled, false otherwise
     */
    private _updateSelectionDepositorAndSaveButton(clearDepositor: boolean = false): boolean {
        if (clearDepositor && this._isAllowedDepositorChange) {
            this.selectionDepositor.value = null;
        }
        this._updateSelectionDepositor();
        return this._updateSaveButton();
    }

    /**
     * Update the selectionDepositor
     * @returns true if the selectionDepositor is enabled, false otherwise
     */
    private _updateSelectionDepositor(): boolean {
        this.selectionDepositor.isDisabled = !this._isAllowedDepositorChange || !this.selectionSite.value;
        return !this.selectionDepositor.isDisabled;
    }

    /**
     * Check if the site and depositor have values
     * @returns true if the site and depositor have values, false otherwise
     */
    private _checkSiteDepositorHasValues(): boolean {
        return !!(this.selectionSite.value?.site?.code && this.selectionDepositor.value?.code);
    }

    /**
     * Update the saveButton
     * @param disableButton
     * @returns true if the saveButton is enabled, false otherwise
     */
    private _updateSaveButton(disableButton: boolean = false): boolean {
        this.saveButton.isDisabled = disableButton || !this._checkSiteDepositorHasValues();
        return !this.saveButton.isDisabled;
    }

    /**
     * Update the site, depositor selection fields, storage, and, optionally, the menu.
     * @param selectedSiteDepositor
     * @param isUpdateMenu
     */
    private async _updateSelectionFieldsSiteDepositorAndStorage(
        selectedSiteDepositor: SiteDepositor | undefined,
        isUpdateMenu: boolean = true,
    ) {
        // Update the site and depositor selection fields
        await this._setMobileSettings(selectedSiteDepositor, isUpdateMenu);
        // Set the placeholder to the current Set Site for better user experience
        this.selectionSite.placeholder = selectedSiteDepositor?.site;
        // Set the placeholder to the current depositor Site for better user experience
        this.selectionDepositor.placeholder = selectedSiteDepositor?.depositor;
        this.selectionDepositor.isReadOnly = !this._isAllowedDepositorChange;
        this.selectionSite.isReadOnly = !this._isAllowedSiteChange;
        this.selectionSite.isDisabled = !this._isAllowedSiteChange;
    }

    /**
     * Technical attributes
     */
    private _isAllowedSiteChange = false;

    private _isAllowedDepositorChange = false;
}
