/**
 * Client functions for user extension : temporary functions before server-side implementation
 */
import type { Depositor } from '@sage/wh-master-data-api';
import type { Site, User } from '@sage/wh-system-api';
import { getDefaultSiteFromModule, getUserProfile } from '@sage/wh-system/lib/client-functions/user-parameters-values';

/**
 * get default site for the user for GEODE module
 * @param user
 * @returns default site for the user
 */
export function getDefaultSite(user: User): Site | null {
    return getUserProfile(user) ? getDefaultSiteFromModule(user, 2) : null;
}

/**
 * Read site from given site
 * Only instance with property code
 * @param pageInstance
 * @param siteCode
 * @param depositorCode
 * @returns site when exist
 */
export async function readSite(pageInstance: any, siteCode: string): Promise<Site | null> {
    try {
        return (await pageInstance.$.graph
            .node('@sage/wh-system/Site')
            .read(
                {
                    code: true,
                },
                `${siteCode?.toUpperCase()}`,
            )
            .execute()) as Site | null;
    } catch (_error) {
        return null;
    }
}

/**
 * Read depositor from given site and depositor codes
 * Only instance with property site and code
 * @param pageInstance
 * @param siteCode
 * @param depositorCode
 * @returns depositor when exist
 */
export async function readDepositor(
    pageInstance: any,
    siteCode: string,
    depositorCode: string,
): Promise<Depositor | null> {
    try {
        return (await pageInstance.$.graph
            .node('@sage/wh-master-data/Depositor')
            .read(
                {
                    site: {
                        code: true,
                    },
                    code: true,
                },
                `${siteCode?.toUpperCase()}|${depositorCode?.toUpperCase()}`,
            )
            .execute()) as Depositor | null;
    } catch (_error) {
        return null;
    }
}

/**
 *  Read user from given user code for print destinations and site depositor
 * @param pageInstance
 * @param userCode
 * @returns
 */
export async function readUser(pageInstance: any, userCode?: string): Promise<User | null> {
    try {
        return (await pageInstance.$.graph
            .node('@sage/wh-system/User')
            .read(
                {
                    code: true,
                    name: true,
                    functionProfile: {
                        profile: true,
                        userProfileDefaultSites: {
                            query: {
                                edges: {
                                    node: {
                                        site: {
                                            code: true,
                                        },
                                    },
                                },
                            },
                        },
                    },
                    profession: {
                        profession: true,
                        functionProfile: {
                            profile: true,
                            userProfileDefaultSites: {
                                query: {
                                    edges: {
                                        node: {
                                            site: {
                                                code: true,
                                            },
                                        },
                                    },
                                },
                            },
                        },
                    },
                    userDestinations: {
                        query: {
                            edges: {
                                node: {
                                    code: true,
                                    destination: {
                                        code: true,
                                        description: true,
                                        destination: true,
                                    },
                                },
                            },
                        },
                    },
                    currentDepositorCode: true,
                    currentSiteCode: true,
                    defaultSiteCode: true,
                    isAllowedDepositorChange: true,
                    isAllowedSiteChange: true,
                },
                `${userCode?.toUpperCase() ?? pageInstance.$.userCode?.toUpperCase()}`,
            )
            .execute()) as User | null;
    } catch (_error) {
        return null;
    }
}
