import * as ui from '@sage/xtrem-ui';
// import { Storage } from '@sage/xtrem-shared';
type Storage = any; // Temporary fix until Storage type is exported from @sage/xtrem-shared

/**
 *  Selected site and depositor type
 */
export type SiteDepositor = { site: string; depositor?: string };
export type LabelDocumentDestination = { labelDestination?: string; documentDestination?: string };

/**
 *  Get the selected site and depositor
 * @param pageInstance
 * @returns
 */
export function getSelectedSiteDepositor(pageInstance: any): SiteDepositor | undefined {
    const siteDepositor = pageInstance?.$.storage.get('mobile-selected-site-depositor');
    return siteDepositor ? JSON.parse(siteDepositor) : undefined;
}

/**
 *  Get the selected site and depositor by storage
 * @param storage
 * @returns
 */
export function getSelectedSiteDepositorByStorage(storage: Storage): SiteDepositor | undefined {
    const siteDepositor = storage?.get('mobile-selected-site-depositor');
    return siteDepositor ? JSON.parse(siteDepositor) : undefined;
}

/**
 *  Get the selected site
 * @param pageInstance
 * @returns
 */
export function getSelectedSite(pageInstance: any): string | undefined {
    return getSelectedSiteDepositor(pageInstance)?.site;
}

/**
 *  Get the selected site by storage
 * @param storage
 * @returns
 */
export function getSelectedSiteByStorage(storage: Storage): string | undefined {
    return getSelectedSiteDepositorByStorage(storage)?.site;
}

/**
 *  Get the selected depositor
 * @param pageInstance
 * @returns
 */
export function getSelectedDepositor(pageInstance: any): string | undefined {
    return getSelectedSiteDepositor(pageInstance)?.depositor;
}

/**
 *  Get the selected depositor by storage
 * @param storage
 * @returns
 */
export function getSelectedDepositorByStorage(storage: Storage): string | undefined {
    return getSelectedSiteDepositorByStorage(storage)?.depositor;
}

/**
 * Set the selected site and depositor
 * @param pageInstance
 * @param site
 * @param depositor
 */
export function setSelectedSiteDepositorByCode(pageInstance: any, site?: string, depositor?: string): void {
    if (site) {
        setSelectedSiteDepositor(pageInstance, <SiteDepositor>{
            site,
            ...(depositor && { depositor }),
        });
    } else {
        removeSelectedSiteDepositor(pageInstance);
    }
}

/**
 * Set the selected site and depositor
 * @param pageInstance
 * @param siteDepositor
 */
export function setSelectedSiteDepositor(pageInstance: any, siteDepositor?: SiteDepositor): void {
    if (siteDepositor) {
        pageInstance.$.storage.set('mobile-selected-site-depositor', JSON.stringify(siteDepositor));
    } else {
        removeSelectedSiteDepositor(pageInstance);
    }
}

/**
 * Remove the selected site and depositor
 * @param pageInstance
 */
export function removeSelectedSiteDepositor(pageInstance: any): void {
    pageInstance.$.storage.remove('mobile-selected-site-depositor');
}

/**
 * Set the selected site
 * @param pageInstance
 * @param site
 */
export function setSelectedSite(pageInstance: any, site?: string): void {
    setSelectedSiteDepositor(pageInstance, site ? <SiteDepositor>{ site } : undefined);
}

/**
 * Set the selected depositor
 * @param pageInstance
 * @param depositor
 */
export function setSelectedDepositor(pageInstance: any, depositor: string): void {
    setSelectedSiteDepositor(pageInstance, <SiteDepositor>{
        site: getSelectedSite(pageInstance),
        ...(depositor && { depositor }),
    });
}

/**
 * Save the selected site and depositor to X3
 * @param pageInstance
 * @param siteDepositor
 * @returns
 */
export async function saveSelectedSiteDepositor(
    pageInstance: any,
    user: string | undefined | null,
    siteDepositor: SiteDepositor,
): Promise<boolean> {
    if (siteDepositor?.site && siteDepositor?.depositor) {
        const _args = {
            user: user ?? '',
            site: siteDepositor.site,
            depositor: siteDepositor.depositor,
            errorCode: -1,
        };
        try {
            const response = await pageInstance.$.graph
                .node('@sage/wh-system/UserParameterValue')
                .mutations.updateUserSiteDepositor(
                    {
                        errorCode: true,
                    },
                    { parameters: _args },
                )
                .execute();
            // Check if the response is valid (API error)
            if (response?.errorCode === undefined || response?.errorCode === -1) {
                return false;
            }

            // Check if error occurred)
            return !response.errorCode;
        } catch (error) {
            ui.console.error('Error saving selected site and depositor', (<Error>error)?.message);
        }
    }
    return false;
}

/**
 * Get the label and document destinations
 * @param pageInstance
 * @returns
 */
export function getLabelDocumentDestination(pageInstance: any): LabelDocumentDestination | undefined {
    const labelDocumentDestination = pageInstance.$.storage.get('mobile-label-document-destination');
    return labelDocumentDestination ? JSON.parse(labelDocumentDestination) : undefined;
}

/**
 * Set the label and document destinations
 * @param pageInstance
 * @param labelDocumentDestination
 */
export function setLabelDocumentDestination(
    pageInstance: any,
    labelDocumentDestination?: LabelDocumentDestination,
): void {
    if (
        labelDocumentDestination &&
        (labelDocumentDestination.labelDestination || labelDocumentDestination.documentDestination)
    ) {
        pageInstance.$.storage.set(
            'mobile-label-document-destination',
            JSON.stringify(<LabelDocumentDestination>{
                ...(labelDocumentDestination.labelDestination && {
                    labelDestination: labelDocumentDestination.labelDestination,
                }),
                ...(labelDocumentDestination.documentDestination && {
                    documentDestination: labelDocumentDestination.documentDestination,
                }),
            }),
        );
    } else {
        removeLabelDocumentDestination(pageInstance);
    }
}

/**
 * Set the label and document destinations by codes
 * @param pageInstance
 * @returns
 */
export function setLabelDocumentDestinationCode(
    pageInstance: any,
    labelDestination?: string,
    documentDestination?: string,
): void {
    if (labelDestination || documentDestination) {
        setLabelDocumentDestination(pageInstance, <LabelDocumentDestination>{
            ...(labelDestination && { labelDestination }),
            ...(documentDestination && { documentDestination }),
        });
    } else {
        removeLabelDocumentDestination(pageInstance);
    }
}

/**
 * Remove the label and document destinations
 * @param pageInstance
 */
export function removeLabelDocumentDestination(pageInstance: any): void {
    pageInstance.$.storage.remove('mobile-label-document-destination');
}

/**
 * Get the label destination
 * @param pageInstance
 * @returns
 */
export function getLabelDestination(pageInstance: any): string | undefined {
    return getLabelDocumentDestination(pageInstance)?.labelDestination;
}

/**
 * Get the document destination
 * @param pageInstance
 * @returns
 */
export function getDocumentDestination(pageInstance: any): string | undefined {
    return getLabelDocumentDestination(pageInstance)?.documentDestination;
}

/**
 * Set the label destination
 * @param pageInstance
 * @param labelDestination
 */
export function setLabelDestination(pageInstance: any, labelDestination?: string): void {
    setLabelDocumentDestination(pageInstance, <LabelDocumentDestination>{
        labelDestination,
        documentDestination: getLabelDocumentDestination(pageInstance)?.documentDestination,
    });
}

/**
 * Set the document destination
 * @param pageInstance
 * @param documentDestination
 */
export function setDocumentDestination(pageInstance: any, documentDestination?: string): void {
    setLabelDocumentDestination(pageInstance, <LabelDocumentDestination>{
        labelDestination: getLabelDocumentDestination(pageInstance)?.labelDestination,
        documentDestination,
    });
}
