import * as ui from '@sage/xtrem-ui';
import { dialogMessage } from './dialogs';
import { SiteDepositor, getSelectedSiteDepositor } from './storage-properties';

/**
 * Return selected stock site for current application
 * @param pageInstance Current page instance
 * @param errorTitle error title when no default site exist
 * @param errorMessage error message when no default site exist
 * @returns default site and depositor or null
 */
export async function getCurrentSiteDepositor(
    pageInstance: ui.Page,
    errorTitle: string,
    errorMessage: string,
): Promise<SiteDepositor | null> {
    const _selectedSiteDepositor = getSelectedSiteDepositor(pageInstance);
    if (!_selectedSiteDepositor?.site) {
        // Please note that there asynchronous processing is not prefixed by an await on purpose
        // so that it can be processed in parallel and not block the main process.
        // Be careful not to leave the current page before the user has closed this dialog box.
        await pageInstance.$.sound.error();
        await dialogMessage(pageInstance, 'error', errorTitle, errorMessage);
        return null;
    }
    return _selectedSiteDepositor;
}

/**
 * Return selected site for current application
 * @param pageInstance Current page instance
 * @param errorTitle error title when no default site exist
 * @param errorMessage error message when no default site exist
 * @returns default site or null
 */
export async function getCurrentSite(
    pageInstance: ui.Page,
    errorTitle: string,
    errorMessage: string,
): Promise<string | null> {
    const _selectedSiteDepositor = await getCurrentSiteDepositor(pageInstance, errorTitle, errorMessage);
    return _selectedSiteDepositor?.site ?? null;
}

/**
 * Return current depositor for current application
 * @param pageInstance Current page instance
 * @param errorTitle error title when no default site exist
 * @param errorMessage error message when no default site exist
 * @returns default depositor or null
 */
export async function getCurrentDepositor(
    pageInstance: ui.Page,
    errorTitle: string,
    errorMessage: string,
): Promise<string | null> {
    const _selectedSiteDepositor = await getCurrentSiteDepositor(pageInstance, errorTitle, errorMessage);
    return _selectedSiteDepositor?.depositor ?? null;
}
