import type { Depositor } from '@sage/wh-master-data-api';
import { extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { getSelectedSiteDepositor } from './storage-properties';

/**
 *  Depositor configuration interface must be located in the same folder as the client function
 *  Duplicated in environnement interfaces
 */
export interface DepositorConfiguration {
    isLotNumber: boolean;
    isSupportNumber: boolean;
}

/**
 *  Load current depositor configuration
 * @param pageInstance
 *  @returns
 */
export async function getDepositorConfiguration(pageInstance: ui.Page): Promise<DepositorConfiguration | undefined> {
    const _siteDepositorCode = getSelectedSiteDepositor(pageInstance);
    const _siteCode = _siteDepositorCode?.site;
    const _depositorCode = _siteDepositorCode?.depositor;

    interface ExtractEdgesDepositorInterface {
        code: string;
        isLotNumber: boolean;
        isSupportNumber: boolean;
        site: { code: string };
    }
    try {
        if (_siteCode && _depositorCode) {
            const _response = extractEdges<ExtractEdgesDepositorInterface>(
                await pageInstance.$.graph
                    .node('@sage/wh-master-data/Depositor')
                    .query(
                        ui.queryUtils.edgesSelector<Depositor>(
                            {
                                code: true,
                                isLotNumber: true,
                                isSupportNumber: true,
                            },
                            {
                                filter: {
                                    site: { code: _siteCode },
                                    code: _depositorCode,
                                },
                            },
                        ),
                    )
                    .execute(),
            )[0];

            if (_response) {
                return <DepositorConfiguration>{
                    isLotNumber: _response.isLotNumber,
                    isSupportNumber: _response.isSupportNumber,
                };
            }
        }
    } catch (error) {
        ui.console.error(`Error reading depositor configuration :\n${error}`);
    }
    return undefined;
}
