"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDefaultSiteCode = getDefaultSiteCode;
exports.getCurrentSiteCode = getCurrentSiteCode;
exports.getCurrentDepositorCode = getCurrentDepositorCode;
exports.getByUserIsAllowedSiteChange = getByUserIsAllowedSiteChange;
exports.getByUserIsAllowedDepositorChange = getByUserIsAllowedDepositorChange;
exports.getByUserCurrentSiteCode = getByUserCurrentSiteCode;
exports.getByUserCurrentDepositorCode = getByUserCurrentDepositorCode;
exports.getIsUserAllowedSiteChange = getIsUserAllowedSiteChange;
exports.getIsUserAllowedDepositorChange = getIsUserAllowedDepositorChange;
exports.readSite = readSite;
exports.readUser = readUser;
exports.readDepositor = readDepositor;
const whSystem = require("@sage/wh-system");
const xtrem_core_1 = require("@sage/xtrem-core");
const x3SystemUtils = require("@sage/xtrem-x3-system-utils");
const whMasterData = require("..");
/**
 * get default site code
 * @param _this
 * @returns default site for current user
 */
async function getDefaultSiteCode(_this) {
    return (await whSystem.functions.getDefaultSiteFromModule(_this, 2))?.code ?? '';
}
/**
 *  get site code for current user
 * @param _this
 * @param user
 * @returns
 */
async function getCurrentSiteCode(_this) {
    return x3SystemUtils.nodes.SysGeneralParameter.readByUserParameterValueString(_this.$.context, {
        code: 'FCY',
        userCode: await _this.code,
    });
}
/**
 * get depositor code for current user
 * @param _this
 * @param user
 * @returns
 */
async function getCurrentDepositorCode(_this) {
    return x3SystemUtils.nodes.SysGeneralParameter.readByUserParameterValueString(_this.$.context, {
        code: 'DEP',
        userCode: await _this.code,
    });
}
/**
 * Read the boolean value of the parameter for current user
 * @param parameterCode
 * @returns
 */
async function _readByUserParameterValueBoolean(_this, parameterCode) {
    return x3SystemUtils.nodes.SysGeneralParameter.readByUserParameterValueBoolean(_this.$.context, {
        code: parameterCode,
        userCode: await _this.code,
    });
}
/**
 * Check if you can enter the site
 * @param _this
 * @returns
 */
async function getByUserIsAllowedSiteChange(_this) {
    const _result = await _readByUserParameterValueBoolean(_this, 'MULFCY');
    return _result;
}
/**
 * Check if you can enter the depositor
 *  @param _this
 * @returns
 */
async function getByUserIsAllowedDepositorChange(_this) {
    const _result = await _readByUserParameterValueBoolean(_this, 'MULDEP');
    return _result;
}
/**
 * get site code for given user
 * @param this
 * @param userCode
 * @returns default site for the user
 */
async function getByUserCurrentSiteCode(_this) {
    return x3SystemUtils.nodes.SysGeneralParameter.readByUserParameterValueString(_this.$.context, {
        code: 'FCY',
        userCode: await _this.code,
    });
}
/**
 * get depositor code for given user
 *  @param this
 * @param user
 * @returns default depositor for the user
 */
async function getByUserCurrentDepositorCode(_this) {
    return x3SystemUtils.nodes.SysGeneralParameter.readByUserParameterValueString(_this.$.context, {
        code: 'DEP',
        userCode: await _this.code,
    });
}
/**
 * Check if the given user can enter the site
 * @param _this
 * @param userCode
 * @returns
 */
async function getIsUserAllowedSiteChange(_this) {
    const _result = await _readByUserParameterValueBoolean(_this, 'MULFCY');
    return _result;
}
/**
 * Check if the given user can enter the depositor
 * @param _this
 * @param userCode
 * @returns
 */
function getIsUserAllowedDepositorChange(_this) {
    return _readByUserParameterValueBoolean(_this, 'MULDEP');
}
/**
 * Read site from given site
 * Only instance with property code
 * @param _this
 * @param siteCode
 * @returns site when exist
 */
async function readSite(_this, siteCode) {
    if (!siteCode) {
        throw new xtrem_core_1.BusinessRuleError('No site code found');
    }
    try {
        return await _this.$.context.tryRead(whSystem.nodes.Site, {
            code: siteCode,
        });
    }
    catch (error) {
        _this.$.context.logger.error(error);
        return Promise.resolve(null);
    }
}
/**
 *  Read user from given user code for print destinations and site depositor
 * @param pageInstance
 * @param userCode
 * @returns
 */
async function readUser(_this, userCode) {
    const _userCode = userCode?.toUpperCase() ?? (await _this.$.context.user)?.userName?.toUpperCase();
    if (!_userCode) {
        throw new xtrem_core_1.BusinessRuleError('No user code found');
    }
    try {
        return await _this.$.context.tryRead(whSystem.nodes.User, {
            code: _userCode,
        });
    }
    catch (error) {
        _this.$.context.logger.error(error);
        return null;
    }
}
/**
 * Read record section
 */
/**
 * Read depositor from given site and depositor codes
 * Only instance with property site and code
 * @param _this
 * @param siteCode
 * @param depositorCode
 * @returns depositor when exist
 */
async function readDepositor(_this, siteCode, depositorCode) {
    if (!siteCode) {
        throw new xtrem_core_1.BusinessRuleError('No site code found');
    }
    if (!depositorCode) {
        throw new xtrem_core_1.BusinessRuleError('No depositor code found');
    }
    try {
        return await _this.$.context.tryRead(whMasterData.nodes.Depositor, {
            site: siteCode?.toUpperCase(),
            code: depositorCode?.toUpperCase(),
        });
    }
    catch (error) {
        _this.$.context.logger.error(error);
        return null;
    }
}
//# sourceMappingURL=user-extension.js.map