"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSelectedSiteDepositor = getSelectedSiteDepositor;
exports.getSelectedSiteDepositorByStorage = getSelectedSiteDepositorByStorage;
exports.getSelectedSite = getSelectedSite;
exports.getSelectedSiteByStorage = getSelectedSiteByStorage;
exports.getSelectedDepositor = getSelectedDepositor;
exports.getSelectedDepositorByStorage = getSelectedDepositorByStorage;
exports.setSelectedSiteDepositorByCode = setSelectedSiteDepositorByCode;
exports.setSelectedSiteDepositor = setSelectedSiteDepositor;
exports.removeSelectedSiteDepositor = removeSelectedSiteDepositor;
exports.setSelectedSite = setSelectedSite;
exports.setSelectedDepositor = setSelectedDepositor;
exports.saveSelectedSiteDepositor = saveSelectedSiteDepositor;
exports.getLabelDocumentDestination = getLabelDocumentDestination;
exports.setLabelDocumentDestination = setLabelDocumentDestination;
exports.setLabelDocumentDestinationCode = setLabelDocumentDestinationCode;
exports.removeLabelDocumentDestination = removeLabelDocumentDestination;
exports.getLabelDestination = getLabelDestination;
exports.getDocumentDestination = getDocumentDestination;
exports.setLabelDestination = setLabelDestination;
exports.setDocumentDestination = setDocumentDestination;
const typesLib = require("@sage/xtrem-decimal");
const ui = require("@sage/xtrem-ui");
/**
 *  Get the selected site and depositor
 * @param pageInstance
 * @returns
 */
function getSelectedSiteDepositor(pageInstance) {
    const siteDepositor = pageInstance?.$.storage.get('mobile-selected-site-depositor');
    return siteDepositor ? JSON.parse(siteDepositor) : undefined;
}
/**
 *  Get the selected site and depositor by storage
 * @param storage
 * @returns
 */
function getSelectedSiteDepositorByStorage(storage) {
    const siteDepositor = storage?.get('mobile-selected-site-depositor');
    return siteDepositor ? JSON.parse(siteDepositor) : undefined;
}
/**
 *  Get the selected site
 * @param pageInstance
 * @returns
 */
function getSelectedSite(pageInstance) {
    return getSelectedSiteDepositor(pageInstance)?.site;
}
/**
 *  Get the selected site by storage
 * @param storage
 * @returns
 */
function getSelectedSiteByStorage(storage) {
    return getSelectedSiteDepositorByStorage(storage)?.site;
}
/**
 *  Get the selected depositor
 * @param pageInstance
 * @returns
 */
function getSelectedDepositor(pageInstance) {
    return getSelectedSiteDepositor(pageInstance)?.depositor;
}
/**
 *  Get the selected depositor by storage
 * @param storage
 * @returns
 */
function getSelectedDepositorByStorage(storage) {
    return getSelectedSiteDepositorByStorage(storage)?.depositor;
}
/**
 * Set the selected site and depositor
 * @param pageInstance
 * @param site
 * @param depositor
 */
function setSelectedSiteDepositorByCode(pageInstance, site, depositor) {
    if (site) {
        setSelectedSiteDepositor(pageInstance, {
            site,
            ...(depositor && { depositor }),
        });
    }
    else {
        removeSelectedSiteDepositor(pageInstance);
    }
}
/**
 * Set the selected site and depositor
 * @param pageInstance
 * @param siteDepositor
 */
function setSelectedSiteDepositor(pageInstance, siteDepositor) {
    if (siteDepositor) {
        pageInstance.$.storage.set('mobile-selected-site-depositor', JSON.stringify(siteDepositor));
    }
    else {
        removeSelectedSiteDepositor(pageInstance);
    }
}
/**
 * Remove the selected site and depositor
 * @param pageInstance
 */
function removeSelectedSiteDepositor(pageInstance) {
    pageInstance.$.storage.remove('mobile-selected-site-depositor');
}
/**
 * Set the selected site
 * @param pageInstance
 * @param site
 */
function setSelectedSite(pageInstance, site) {
    setSelectedSiteDepositor(pageInstance, site ? { site } : undefined);
}
/**
 * Set the selected depositor
 * @param pageInstance
 * @param depositor
 */
function setSelectedDepositor(pageInstance, depositor) {
    setSelectedSiteDepositor(pageInstance, {
        site: getSelectedSite(pageInstance),
        ...(depositor && { depositor }),
    });
}
/**
 * Save the selected site and depositor to X3
 * @param pageInstance
 * @param siteDepositor
 * @returns
 */
async function saveSelectedSiteDepositor(pageInstance, user, siteDepositor) {
    if (siteDepositor?.site && siteDepositor?.depositor) {
        const _args = {
            user: user ?? '',
            site: siteDepositor.site,
            depositor: siteDepositor.depositor,
            errorCode: -1,
        };
        try {
            const response = await pageInstance.$.graph
                .node('@sage/wh-system/UserParameterValue')
                .mutations.updateUserSiteDepositor({
                errorCode: true,
            }, { parameters: _args })
                .execute();
            // Check if the response is valid (API error)
            if (typesLib.strictEq(response?.errorCode, undefined) || typesLib.strictEq(response?.errorCode, -1)) {
                return false;
            }
            // Check if error occurred)
            return !response.errorCode;
        }
        catch (error) {
            ui.console.error('Error saving selected site and depositor', error?.message);
        }
    }
    return false;
}
/**
 * Get the label and document destinations
 * @param pageInstance
 * @returns
 */
function getLabelDocumentDestination(pageInstance) {
    const labelDocumentDestination = pageInstance.$.storage.get('mobile-label-document-destination');
    return labelDocumentDestination ? JSON.parse(labelDocumentDestination) : undefined;
}
/**
 * Set the label and document destinations
 * @param pageInstance
 * @param labelDocumentDestination
 */
function setLabelDocumentDestination(pageInstance, labelDocumentDestination) {
    if (labelDocumentDestination &&
        (labelDocumentDestination.labelDestination || labelDocumentDestination.documentDestination)) {
        pageInstance.$.storage.set('mobile-label-document-destination', JSON.stringify({
            ...(labelDocumentDestination.labelDestination && {
                labelDestination: labelDocumentDestination.labelDestination,
            }),
            ...(labelDocumentDestination.documentDestination && {
                documentDestination: labelDocumentDestination.documentDestination,
            }),
        }));
    }
    else {
        removeLabelDocumentDestination(pageInstance);
    }
}
/**
 * Set the label and document destinations by codes
 * @param pageInstance
 * @returns
 */
function setLabelDocumentDestinationCode(pageInstance, labelDestination, documentDestination) {
    if (labelDestination || documentDestination) {
        setLabelDocumentDestination(pageInstance, {
            ...(labelDestination && { labelDestination }),
            ...(documentDestination && { documentDestination }),
        });
    }
    else {
        removeLabelDocumentDestination(pageInstance);
    }
}
/**
 * Remove the label and document destinations
 * @param pageInstance
 */
function removeLabelDocumentDestination(pageInstance) {
    pageInstance.$.storage.remove('mobile-label-document-destination');
}
/**
 * Get the label destination
 * @param pageInstance
 * @returns
 */
function getLabelDestination(pageInstance) {
    return getLabelDocumentDestination(pageInstance)?.labelDestination;
}
/**
 * Get the document destination
 * @param pageInstance
 * @returns
 */
function getDocumentDestination(pageInstance) {
    return getLabelDocumentDestination(pageInstance)?.documentDestination;
}
/**
 * Set the label destination
 * @param pageInstance
 * @param labelDestination
 */
function setLabelDestination(pageInstance, labelDestination) {
    setLabelDocumentDestination(pageInstance, {
        labelDestination,
        documentDestination: getLabelDocumentDestination(pageInstance)?.documentDestination,
    });
}
/**
 * Set the document destination
 * @param pageInstance
 * @param documentDestination
 */
function setDocumentDestination(pageInstance, documentDestination) {
    setLabelDocumentDestination(pageInstance, {
        labelDestination: getLabelDocumentDestination(pageInstance)?.labelDestination,
        documentDestination,
    });
}
//# sourceMappingURL=storage-properties.js.map