"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizedOperation = void 0;
exports.getAuthorizedOperator = getAuthorizedOperator;
exports.isOperatorRequired = isOperatorRequired;
exports.isAuthorizedOperator = isAuthorizedOperator;
const typesLib = require("@sage/xtrem-decimal");
const xtrem_client_1 = require("@sage/xtrem-client");
const ui = require("@sage/xtrem-ui");
const storage_properties_1 = require("./storage-properties");
/**
 * @description: This enum is used to define the authorized operation for the current site.
 * @example: AuthorizedOperation.receipt
 */
var AuthorizedOperation;
(function (AuthorizedOperation) {
    AuthorizedOperation["adjustment"] = "adjustment";
    AuthorizedOperation["carrierEdi"] = "carrierEdi";
    AuthorizedOperation["consumableInput"] = "consumableInput";
    AuthorizedOperation["consumableOutput"] = "consumableOutput";
    AuthorizedOperation["picking"] = "picking";
    AuthorizedOperation["receipt"] = "receipt";
    AuthorizedOperation["repacking"] = "repacking";
    AuthorizedOperation["simplifiedStockCount"] = "simplifiedStockCount";
    AuthorizedOperation["transfer"] = "transfer";
})(AuthorizedOperation || (exports.AuthorizedOperation = AuthorizedOperation = {}));
/**
 *  @description: This function is used to get the authorized operator for current site:
 *                she returns an operator code when required, otherwise an empty string, and undefined when not applicable.
 *  @example: getAuthorizedOperator(AuthorizedOperation.receipt)
 *  @param pageInstance: ui.Page,
 *  @param operatorCode : string
 *  @param operationCode : AuthorizedOperation
 *  @returns: Promise<string | undefined | Error>
 */
async function getAuthorizedOperator(pageInstance, operationCode) {
    const _siteCode = (0, storage_properties_1.getSelectedSite)(pageInstance);
    if (typesLib.strictEq(_siteCode, undefined)) {
        ui.console.error('Missing site code error');
        return undefined;
    }
    // Check if the operator is required for the current site and operation
    const _isOperatorRequired = await isOperatorRequired(pageInstance, operationCode);
    if (!_isOperatorRequired || _isOperatorRequired instanceof Error) {
        return _isOperatorRequired instanceof Error ? _isOperatorRequired : '';
    }
    let _operatorCode;
    try {
        const options = {
            fullScreen: true,
        };
        // If the window is closed without validation, we catch the standard exception and return undefined
        _operatorCode = await pageInstance.$.dialog
            .page('@sage/wh-master-data/MobileGetOperator', { getOperatorArgs: _getOperatorParameters(_siteCode, operationCode) }, options)
            .catch(() => '');
    }
    catch (error) {
        ui.console.error('Error reading operator:\n', JSON.stringify(error));
    }
    if (!_operatorCode) {
        return undefined;
    }
    // Check if the operator is authorized for the current site and operation
    const _operatorAuthorized = await isAuthorizedOperator(pageInstance, _operatorCode, operationCode);
    // Return undefined if the operator is not authorized
    if (_operatorAuthorized instanceof Error) {
        return _operatorAuthorized;
    }
    return _operatorAuthorized ? _operatorCode : undefined;
}
/**
 *  @description: This function is used to get the operator parameters for the current site and operation.
 *  @example: _getOperatorParameters(this._siteSelectedCode, AuthorizedOperation.receipt)
 *  @param siteCode : string
 *  @param operationCode : AuthorizedOperation
 *  @returns: getOperatorArgs: string
 */
function _getOperatorParameters(siteCode, operationCode) {
    return JSON.stringify({ operationCode, siteCode });
}
/**
 *  @description: This function is used to get the authorized operator for current site :
 *                she valid an operator code, and undefined when not applicable (no standard).
 *  @example: isOperatorRequired(this, this._siteSelectedCode, AuthorizedOperation.receipt)
 *  @param pageInstance: ui.Page,
 *  @param operationCode : AuthorizedOperation
 *  @returns: Promise<boolean | error>
 */
async function isOperatorRequired(pageInstance, operationCode) {
    const _siteCode = (0, storage_properties_1.getSelectedSite)(pageInstance);
    if (typesLib.strictEq(_siteCode, undefined)) {
        ui.console.error('Missing site code error');
        return false;
    }
    try {
        const _response = (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
            .node('@sage/whMasterData/whSite')
            .query(ui.queryUtils.edgesSelector({
            isAdjustementOperationCodeAllowed: true,
            isConsumableInputControlCodeAllowed: true,
            isConsumableOutputControlCodeAllowed: true,
            isEdiOperationCode: true,
            isPickingOperationCodeAllowed: true,
            isReceiptOperationCodeAllowed: true,
            isSimplifiedStockCountCodeAllowed: true,
            isRepackingOperationCodeAllowed: true,
            isTransferOperationCodeAllowed: true,
        }, {
            filter: {
                site: { code: _siteCode },
            },
        }))
            .execute());
        if (_response.length) {
            const _site = _response[0];
            switch (operationCode) {
                case AuthorizedOperation.adjustment:
                    return _site.isAdjustementOperationCodeAllowed;
                case AuthorizedOperation.carrierEdi:
                    return _site.isEdiOperationCode;
                case AuthorizedOperation.consumableInput:
                    return _site.isConsumableInputControlCodeAllowed;
                case AuthorizedOperation.consumableOutput:
                    return _site.isConsumableOutputControlCodeAllowed;
                case AuthorizedOperation.picking:
                    return _site.isPickingOperationCodeAllowed;
                case AuthorizedOperation.receipt:
                    return _site.isReceiptOperationCodeAllowed;
                case AuthorizedOperation.repacking:
                    return _site.isRepackingOperationCodeAllowed;
                case AuthorizedOperation.simplifiedStockCount:
                    return _site.isSimplifiedStockCountCodeAllowed;
                case AuthorizedOperation.transfer:
                    return _site.isTransferOperationCodeAllowed;
                default:
                    return false;
            }
        }
    }
    catch (error) {
        ui.console.error('Error reading product container:\n', JSON.stringify(error));
        return error;
    }
    return false;
}
/**
 *  @description: This function is used to get the authorized operator for current site :
 *                she valid an operator code, and undefined when not applicable (no standard).
 *  @example: isAuthorizedOperator(this, this._siteSelectedCode, _operatorCode, AuthorizedOperation.receipt)
 *  @param pageInstance: ui.Page,
 *  @param operatorCode : string
 *  @param operationCode : AuthorizedOperation
 *  @returns: Promise<boolean | error>
 */
async function isAuthorizedOperator(pageInstance, operatorCode, operationCode) {
    const _siteCode = (0, storage_properties_1.getSelectedSite)(pageInstance);
    if (typesLib.strictEq(_siteCode, undefined)) {
        ui.console.error('Missing site code error');
        return false;
    }
    try {
        const _response = (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
            .node('@sage/whMasterData/Operand')
            .query(ui.queryUtils.edgesSelector({
            isCarrierEdiAuthorized: true,
            isConsumableInputAuthorized: true,
            isConsumableOuputAuthorized: true,
            isManualAdjustmentAuthorized: true,
            isPickingAuthorized: true,
            isReceiptAuthorized: true,
            isRepackingAuthorized: true,
            isSimplifiedStockCountAuthorized: true,
            isTransferAuthorized: true,
        }, {
            filter: {
                site: { code: _siteCode },
                code: operatorCode,
            },
        }))
            .execute());
        if (_response.length) {
            const _operator = _response[0];
            switch (operationCode) {
                case AuthorizedOperation.adjustment:
                    return _operator.isManualAdjustmentAuthorized;
                case AuthorizedOperation.carrierEdi:
                    return _operator.isCarrierEdiAuthorized;
                case AuthorizedOperation.consumableInput:
                    return _operator.isConsumableInputAuthorized;
                case AuthorizedOperation.consumableOutput:
                    return _operator.isConsumableOuputAuthorized;
                case AuthorizedOperation.picking:
                    return _operator.isPickingAuthorized;
                case AuthorizedOperation.receipt:
                    return _operator.isReceiptAuthorized;
                case AuthorizedOperation.repacking:
                    return _operator.isRepackingAuthorized;
                case AuthorizedOperation.simplifiedStockCount:
                    return _operator.isSimplifiedStockCountAuthorized;
                case AuthorizedOperation.transfer:
                    return _operator.isTransferAuthorized;
                default:
                    return false;
            }
        }
    }
    catch (error) {
        ui.console.error('Error reading product container:\n', JSON.stringify(error));
        return error;
    }
    return false;
}
//# sourceMappingURL=get-authorized-operator.js.map