"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const client_dynamodb_1 = require("@aws-sdk/client-dynamodb");
class DynamoDBTenantAppDatasource {
    db;
    tableName;
    constructor(tableName, configOverride) {
        this.db = new client_dynamodb_1.DynamoDB(configOverride);
        this.tableName = tableName;
    }
    async getAppsOfTenant(tenantId) {
        let lastEvaluatedKey = undefined;
        const responseMap = {};
        do {
            const getAppsQuery = {
                TableName: this.tableName,
                KeyConditionExpression: "PK = :pk and begins_with(SK, :skPrefix)",
                ExpressionAttributeValues: {
                    ":pk": { S: `TNT#${tenantId}` },
                    ":skPrefix": { S: "APP#" },
                },
                ExclusiveStartKey: lastEvaluatedKey
            };
            const queryResult = await this.db.query(getAppsQuery);
            lastEvaluatedKey = queryResult.LastEvaluatedKey;
            if (queryResult.Items) {
                queryResult.Items.map(DynamoDBTenantAppDatasource.fromDynamoDb).reduce((previousValue, currentValue) => {
                    if (currentValue) {
                        if (Object.prototype.hasOwnProperty.call(previousValue, currentValue.cluster)) {
                            previousValue[currentValue.cluster].push(currentValue.appCode);
                        }
                        else {
                            previousValue[currentValue.cluster] = [currentValue.appCode];
                        }
                    }
                    return previousValue;
                }, responseMap);
            }
        } while (lastEvaluatedKey);
        return responseMap;
    }
    async getEmailDomainOfTenant(tenantId) {
        const tenant = await this.getTenant(tenantId);
        const customer = await this.getCustomer(tenant.customerId);
        return customer.emailDomain;
    }
    async getCustomer(customerId) {
        if (!customerId) {
            return {};
        }
        const customerCmdOutput = await this.db.getItem({ Key: { PK: { S: `CST#${customerId}` }, SK: { S: "CUSTOMER" } }, TableName: this.tableName });
        if (!customerCmdOutput.Item) {
            return {};
        }
        if (customerCmdOutput.Item.emailDomain && customerCmdOutput.Item.emailDomain.S) {
            return {
                id: customerCmdOutput.Item.PK.S.substring(4),
                emailDomain: customerCmdOutput.Item.emailDomain.S
            };
        }
        return {
            id: customerCmdOutput.Item.PK.S.substring(4)
        };
    }
    async getTenant(tenantId) {
        const tenantCmdOutput = await this.db.getItem({ Key: { PK: { S: `TNT#${tenantId}` }, SK: { S: "TENANT" } }, TableName: this.tableName });
        if (!tenantCmdOutput.Item) {
            return {};
        }
        return {
            id: tenantCmdOutput.Item.PK.S.substring(4),
            customerId: tenantCmdOutput.Item.customerId.S
        };
    }
    static fromDynamoDb = (tenantAppRow) => {
        const rawApp = tenantAppRow.SK.S?.split('#'); // ex: APP#sdmo
        let appCode = undefined;
        if (rawApp && rawApp.length === 2) {
            appCode = rawApp[1];
        }
        else {
            return null;
        }
        if (!tenantAppRow.enabled.BOOL) {
            return null;
        }
        if (tenantAppRow.status.S !== "ready") {
            return null;
        }
        if (!tenantAppRow.cluster.S) {
            return null;
        }
        return {
            cluster: tenantAppRow.cluster.S,
            appCode
        };
    };
}
exports.default = DynamoDBTenantAppDatasource;
//# sourceMappingURL=data:application/json;base64,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