"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CachedTenantAppDataStore = void 0;
const cacheable_1 = require("cacheable");
const MAX_TENANT_IN_CACHE = 1000;
const TENANT_CACHE_TTL_MS = 1000 * 60 * 5;
class CachedTenantAppDataStore {
    tenantCache;
    tenantAppsCache;
    customerCache;
    actualTenantAppDatasource;
    cacheTTLInMS;
    constructor(actualTenantAppDatasource, cacheTTLInMS = TENANT_CACHE_TTL_MS) {
        this.actualTenantAppDatasource = actualTenantAppDatasource;
        this.cacheTTLInMS = cacheTTLInMS;
        this.tenantCache = new cacheable_1.CacheableMemory({ ttl: this.cacheTTLInMS, lruSize: MAX_TENANT_IN_CACHE });
        this.tenantAppsCache = new cacheable_1.CacheableMemory({ ttl: this.cacheTTLInMS, lruSize: MAX_TENANT_IN_CACHE });
        this.customerCache = new cacheable_1.CacheableMemory({ ttl: this.cacheTTLInMS, lruSize: MAX_TENANT_IN_CACHE });
    }
    async getAppsOfTenant(tenantId) {
        const cacheResult = this.tenantAppsCache.get(tenantId);
        if (cacheResult) {
            return cacheResult;
        }
        const appsOfTenant = await this.actualTenantAppDatasource.getAppsOfTenant(tenantId);
        this.tenantAppsCache.set(tenantId, appsOfTenant);
        return appsOfTenant;
    }
    async getEmailDomainOfTenant(tenantId) {
        const tenant = await this.getTenant(tenantId);
        const customer = await this.getCustomer(tenant.customerId);
        return customer.emailDomain;
    }
    async getCustomer(customerId) {
        if (!customerId) {
            return {};
        }
        const customerCacheResult = this.customerCache.get(customerId);
        if (customerCacheResult) {
            return customerCacheResult;
        }
        const customer = await this.actualTenantAppDatasource.getCustomer(customerId);
        this.customerCache.set(customerId, customer);
        return customer;
    }
    async getTenant(tenantId) {
        const tenantCacheResult = this.tenantCache.get(tenantId);
        if (tenantCacheResult) {
            return tenantCacheResult;
        }
        const tenant = await this.actualTenantAppDatasource.getTenant(tenantId);
        this.tenantCache.set(tenantId, tenant);
        return tenant;
    }
}
exports.CachedTenantAppDataStore = CachedTenantAppDataStore;
//# sourceMappingURL=data:application/json;base64,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