"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TenantAppHelper = void 0;
const utils_1 = require("../utils");
const datasource_1 = require("./datasource");
const AppsYmlTenantAppDatasource_1 = require("./datasource/AppsYmlTenantAppDatasource");
const singleton = new Map();
class TenantAppHelper {
    tenantDatasource;
    cachedTenantDatasource;
    currentCluster;
    constructor(tenantRegistry, currentXtremCluster, cacheTTLInMilliSecond) {
        this.tenantDatasource = tenantRegistry;
        this.cachedTenantDatasource = new datasource_1.CachedTenantAppDataStore(tenantRegistry, cacheTTLInMilliSecond);
        this.currentCluster = currentXtremCluster;
    }
    /***
     * Return apps of tenant for a given xtrem cluster
     * @param request if request's cluster is not set we will filter apps on the current cluster
     */
    async getAppsOfTenant(request) {
        const registryToUse = request.option?.noCache ? this.tenantDatasource : this.cachedTenantDatasource;
        const appsOfTenant = await registryToUse.getAppsOfTenant(request.tenantId);
        const clusterToSearch = request.cluster || this.currentCluster;
        return appsOfTenant[clusterToSearch] || [];
    }
    /***
     * Return a record object of apps code per cluster id for hte given tenant
     * @param request
     */
    async getAppsOfTenantPerCluster(request) {
        const registryToUse = request.option?.noCache ? this.tenantDatasource : this.cachedTenantDatasource;
        return registryToUse.getAppsOfTenant(request.tenantId);
    }
    /***
     * Return an array of application code that represent all the apps the tenants own across all the cluster
     * @param request
     */
    async getAppsCrossCluster(request) {
        const registryToUse = request.option?.noCache ? this.tenantDatasource : this.cachedTenantDatasource;
        const appsOfTenant = await registryToUse.getAppsOfTenant(request.tenantId);
        return Object.keys(appsOfTenant).reduce((accumulator, currentValue) => {
            // normally we should not have same apps in different cluster but let's be defensive
            appsOfTenant[currentValue].forEach((app) => {
                if (!accumulator.includes(app)) {
                    accumulator.push(app);
                }
            });
            return accumulator;
        }, []);
    }
    /***
     * Return email domain of tenant
     * @param request tenant id and options to search
     */
    async getEmailDomainOfTenant(request) {
        const registryToUse = request.option?.noCache ? this.tenantDatasource : this.cachedTenantDatasource;
        return registryToUse.getEmailDomainOfTenant(request.tenantId);
    }
    /***
     * Return TenanAppHelper instance singleton based on param passed
     * @param awsConfig override of aws config, specify region property
     * @param tableName dynamoDB table name, should be read from xtrem config file
     * @param currentCluster cluster id used when searching only apps of the current cluster
     * @param cacheTTLInMilliSecond optional, how long(milliseconds) we keep record in memory cache, default 5 minute
     */
    static getCloudImplementation(awsConfig, tableName, currentCluster, cacheTTLInMilliSecond) {
        const singletonKey = tableName + awsConfig.region + currentCluster;
        let potentialRepo = singleton.get(singletonKey);
        if (!potentialRepo) {
            potentialRepo = new TenantAppHelper(new datasource_1.DynamoDBTenantAppDatasource(tableName, awsConfig), currentCluster, cacheTTLInMilliSecond);
            singleton.set(singletonKey, potentialRepo);
        }
        return potentialRepo;
    }
    /***
     * Return a local developer singleton implementation, using apps.yml to get list of apps of the tenant, assuming the tenant have them all
     * @param appsYamlPath full path to  the apps.yml file
     * @param localClusterName optional what is your cluster name, local by default
     * @param emailDomain optional what is the emailDomain of your tenant, empty by default
     *
     */
    static getLocalDeveloperImplementation(appsYamlPath, localClusterName = "local", emailDomain) {
        if ((0, utils_1.looksToBeInCloudEnv)()) {
            console.error("Looks like you are using a TenantAppHelper.getLocalDeveloperImplementation in a cloud execution, this method is supposed to be used only for local developer and will prevent xtrem to know what apps the tenant have");
        }
        const singletonKey = `${appsYamlPath}${emailDomain || ""}`;
        let potentialRepo = singleton.get(singletonKey);
        if (!potentialRepo) {
            potentialRepo = new TenantAppHelper(new AppsYmlTenantAppDatasource_1.AppsYmlTenantAppDatasource(appsYamlPath, localClusterName, emailDomain), localClusterName);
            singleton.set(singletonKey, potentialRepo);
        }
        return potentialRepo;
    }
}
exports.TenantAppHelper = TenantAppHelper;
//# sourceMappingURL=data:application/json;base64,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