"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var SysUtility_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SysUtility = void 0;
const typesLib = require("@sage/xtrem-decimal");
const xtrem_core_1 = require("@sage/xtrem-core");
const xtrem_x3_gateway_1 = require("@sage/xtrem-x3-gateway");
const xtrem_x3_sql_manager_1 = require("@sage/xtrem-x3-sql-manager");
const logger = xtrem_core_1.Logger.getLogger(__filename, 'sys-utility');
let SysUtility = class SysUtility extends xtrem_core_1.Node {
    static { SysUtility_1 = this; }
    static parsePropertyValue(property, value) {
        if (typesLib.eq(value, null)) {
            return null;
        }
        const type = typesLib.strictEq(property.type, 'reference') ? property.columnType : property.type;
        switch (type) {
            case 'boolean':
                return typesLib.strictEq(Number(value), 2);
            case 'integer':
                return Number(value);
            case 'decimal':
                return String(value);
            case 'string':
                return String(value);
            case 'date':
                return xtrem_core_1.date.parse(String(value), 'en-US', 'YYYYMMDD').toString();
            case 'enum': {
                const enumDatatype = property.dataType;
                return typesLib.strictEq(Number(value), 0) ? null : enumDatatype.stringValue(Number(value));
            }
            default:
                return value;
        }
    }
    static async getMainNodeNameFromTableName(context, tableName) {
        const folderName = xtrem_x3_gateway_1.FolderManager.getFolderName({ context });
        const pool = xtrem_x3_sql_manager_1.PoolManager.getX3Pool(context);
        const sql = `SELECT a.ANODNAM_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('ANODNAM_0', pool.dialect)}
                    FROM ${folderName}.AWRKLNK a
                    WHERE FLGNODE_0 = 2 AND FLGREFAWM_0 = 2 AND TABREF_0 = '${tableName}'`;
        const response = await pool.execute(sql);
        if (Array.isArray(response) && typesLib.gt(response.length, 0)) {
            return response[0].ANODNAM_0 ?? '';
        }
        return '';
    }
    /**
     * Returns the node key using the table name and X3 key.
     * @param context
     * @param tableName
     * @param x3Names
     * @param source
     * @returns
     */
    static async getNodeKeyFromTableName(context, tableName, x3Names, source) {
        const factories = context.application.getAllFactories().filter(f => {
            const externalStorageManager = f.externalStorageManager;
            if (externalStorageManager?.isDenormalized)
                return false;
            return typesLib.strictEq(f.tableName, tableName);
        });
        if (typesLib.gt(factories.length, 0)) {
            let factory = factories[0];
            if (typesLib.gt(factories.length, 1)) {
                // If there are multiple factories, we need to find the main node name
                // If we can't find the main node name, we will use the first factory found
                const mainNodeName = await SysUtility_1.getMainNodeNameFromTableName(context, tableName);
                const mainFactory = factories.find(f => typesLib.strictEq(f.name, mainNodeName));
                if (mainFactory) {
                    factory = mainFactory;
                }
            }
            const keys = Object.keys(x3Names);
            const key = keys.reduce((r, k) => {
                // ADO1_LAN -> ADO1
                // BPCNUM -> BPCNUM
                const keyParts = typesLib.strictEq(source, 'classic') ? k.split('_') : [k];
                const keyColumn = keyParts[typesLib.sub(keyParts.length, 1)];
                const keyProperty = factory.properties.find(p => typesLib.strictEq(p.columnName, keyColumn));
                if (keyProperty) {
                    const value = x3Names[k];
                    r[keyProperty.name] = SysUtility_1.parsePropertyValue(keyProperty, value); // parse the value here
                }
                return r;
            }, {});
            if (typesLib.strictEq(Object.keys(key).length, keys.length)) {
                return {
                    nodeName: factory.fullName,
                    key: JSON.stringify(key),
                };
            }
        }
        return {
            nodeName: '',
            key: '',
        };
    }
    static { this.functionCache = {}; }
    static { this.functionEtag = ''; }
    static { this.functionLastCheck = Date.now(); }
    static async validateFunctionCache(context, folderName) {
        const now = Date.now();
        // Check if the cache is still valid
        // If the cache is older than 1 minute, we need to refresh it
        if (typesLib.gt(typesLib.add(SysUtility_1.functionLastCheck, typesLib.mul(60, 1000)), now)) {
            // Cache is still valid
            return true;
        }
        SysUtility_1.functionLastCheck = now;
        const pool = xtrem_x3_sql_manager_1.PoolManager.getX3Pool(context);
        const etagSql = `SELECT MAX(f.UPDDATTIM_0) ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('UPDDATTIM_0', pool.dialect)}
                            FROM
                                ${folderName}.AFONCTION f
                            UNION SELECT MAX(o.UPDDATTIM_0) ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('UPDDATTIM_0', pool.dialect)}
                            FROM
                                ${folderName}.AOBJET o`;
        const etagResponse = await pool.execute(etagSql);
        const etag = Array.isArray(etagResponse)
            ? etagResponse.map((row) => String(row.UPDDATTIM_0)).join('~')
            : '';
        if (typesLib.strictNe(SysUtility_1.functionEtag, etag)) {
            // Cache is invalidated
            SysUtility_1.functionEtag = etag;
            SysUtility_1.functionCache = {};
            return false;
        }
        return true;
    }
    /**
     * Retrieves the table name from the function code.
     * The function code is used to find the object code, which is then used to find the table name.
     * The result is cached to improve performance.
     * @param context
     * @param folderName
     * @param functionCode
     * @returns
     */
    static async getTableNameFromFunctionCode(context, folderName, functionCode) {
        const cacheKey = `${folderName}-${functionCode}`;
        if ((await SysUtility_1.validateFunctionCache(context, folderName)) && typesLib.ne(SysUtility_1.functionCache[cacheKey], null)) {
            return SysUtility_1.functionCache[cacheKey];
        }
        const pool = xtrem_x3_sql_manager_1.PoolManager.getX3Pool(context);
        const functionSql = `SELECT f.VALPAR_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('VALPAR_0', pool.dialect)}
                            FROM ${folderName}.AFONCTION f
                            WHERE f.CODINT_0 = ${pool.param(0)}`;
        const functionResponse = await pool.execute(functionSql, [functionCode]);
        if (!Array.isArray(functionResponse) || typesLib.strictEq(functionResponse.length, 0)) {
            SysUtility_1.functionCache[cacheKey] = '';
            return SysUtility_1.functionCache[cacheKey];
        }
        // VALPAR_0 value is surrounded by quotes, we need to remove them to get the correct object code
        const objectCode = String(functionResponse[0].VALPAR_0).replaceAll(/['"]/g, '');
        const objectSql = `SELECT o.NOMFIC_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('NOMFIC_0', pool.dialect)}
                            FROM ${folderName}.AOBJET o
                            WHERE o.ABREV_0 = ${pool.param(0)}`;
        const objectResponse = await pool.execute(objectSql, [objectCode]);
        if (Array.isArray(objectResponse) && typesLib.gt(objectResponse.length, 0)) {
            SysUtility_1.functionCache[cacheKey] = objectResponse[0].NOMFIC_0 ?? '';
            return SysUtility_1.functionCache[cacheKey];
        }
        SysUtility_1.functionCache[cacheKey] = '';
        return SysUtility_1.functionCache[cacheKey];
    }
    /**
     * Retrieves the node key using the function code and X3 key.
     * The function code is used to find the table name, which is then used to find the node key.
     * @param context
     * @param functionCode
     * @param x3Names
     * @returns
     */
    static async getClassicFunctionKey(context, functionCode, x3Names) {
        const folderName = xtrem_x3_gateway_1.FolderManager.getFolderName({ context });
        const tableName = await SysUtility_1.getTableNameFromFunctionCode(context, folderName, functionCode);
        if (tableName) {
            return SysUtility_1.getNodeKeyFromTableName(context, tableName, x3Names, 'classic');
        }
        return Promise.resolve({
            nodeName: '',
            key: '',
        });
    }
    static { this.representationCache = {}; }
    static { this.representationEtag = ''; }
    static { this.representationLastCheck = Date.now(); }
    static async validateRepresentationCache(context, folderName) {
        const now = Date.now();
        // Check if the cache is still valid
        // If the cache is older than 1 minute, we need to refresh it
        if (typesLib.gt(typesLib.add(SysUtility_1.representationLastCheck, typesLib.mul(60, 1000)), now)) {
            // Cache is still valid
            return true;
        }
        SysUtility_1.representationLastCheck = now;
        const pool = xtrem_x3_sql_manager_1.PoolManager.getX3Pool(context);
        const etagSql = `SELECT MAX(r.UPDDATTIM_0) ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('UPDDATTIM_0', pool.dialect)}
                            FROM
                                ${folderName}.ASHW r
                            UNION SELECT MAX(c.UPDDATTIM_0) ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('UPDDATTIM_0', pool.dialect)}
                            FROM
                                ${folderName}.ACLASSE c`;
        const etagResponse = await pool.execute(etagSql);
        const etag = Array.isArray(etagResponse)
            ? etagResponse.map((row) => String(row.UPDDATTIM_0)).join('~')
            : '';
        if (typesLib.strictNe(SysUtility_1.representationEtag, etag)) {
            // Cache is invalidated
            SysUtility_1.representationEtag = etag;
            SysUtility_1.representationCache = {};
            return false;
        }
        return true;
    }
    /**
     * Retrieves the table name from the representation code.
     * The representation code is used to find the class code, which is then used to find the table name.
     * @param context
     * @param folderName
     * @param representationCode
     * @returns
     */
    static async getTableNameFromRepresentationCode(context, folderName, representationCode) {
        const cacheKey = `${folderName}-${representationCode}`;
        if ((await SysUtility_1.validateRepresentationCache(context, folderName)) && typesLib.ne(SysUtility_1.representationCache[cacheKey], null)) {
            return SysUtility_1.representationCache[cacheKey];
        }
        const pool = xtrem_x3_sql_manager_1.PoolManager.getX3Pool(context);
        const objectSql = `SELECT c.TABREF_0 ${xtrem_x3_gateway_1.SqlResolver.makeColumnAlias('TABREF_0', pool.dialect)}
                            FROM ${folderName}.ASHW r
                            JOIN ${folderName}.ACLASSE c ON c.CODCLA_0 = r.CODCLA_0
                            WHERE r.CODREP_0 = ${pool.param(0)}`;
        const response = await pool.execute(objectSql, [representationCode]);
        if (Array.isArray(response) && typesLib.gt(response.length, 0)) {
            SysUtility_1.representationCache[cacheKey] = response[0].TABREF_0 ?? '';
            return SysUtility_1.representationCache[cacheKey];
        }
        SysUtility_1.representationCache[cacheKey] = '';
        return SysUtility_1.representationCache[cacheKey];
    }
    /**
     * Retrieves the node key using the representation code and X3 key.
     * The representation and class code is used to find the table name, which is then used to find the node key.
     * @param context
     * @param representationCode
     * @param x3Names
     * @returns
     */
    static async getRepresentationKey(context, representationCode, x3Names) {
        const folderName = xtrem_x3_gateway_1.FolderManager.getFolderName({ context });
        const tableName = await SysUtility_1.getTableNameFromRepresentationCode(context, folderName, representationCode);
        if (tableName) {
            return SysUtility_1.getNodeKeyFromTableName(context, tableName, x3Names, 'representation');
        }
        return Promise.resolve({
            nodeName: '',
            key: '',
        });
    }
    /**
     * Returns the node key using the function code or representation code and X3 key.
     * @param context
     * @param input
     * @returns
     */
    static parseKey(context, input) {
        // Example input classic function:
        // {
        //     "function": "GESADO",
        //     "name": "OADO",
        //     "x3Names": {
        //         "ADO1_LAN": "BRI",
        //         "ADO1_TYP": "AFC",
        //         "ADO1_COD": "GESADS",
        //         "ADO1_LEV": "10",
        //         "ADO1_SUBLEV": "20"
        //     }
        // }
        //
        // Example input representation function:
        // {
        //     "representation": "BPCUSTOMER",
        //     "x3Names": {
        //         "BPCNUM": "0_EXCLUNIM1"
        //     }
        // }
        const inputData = JSON.parse(input);
        if (inputData.function && inputData.x3Names) {
            return SysUtility_1.getClassicFunctionKey(context, inputData.function, inputData.x3Names).catch(error => {
                logger.verbose(() => `Error parsing function key: ${inputData.function}: ${error}`);
                return {
                    nodeName: '',
                    key: '',
                };
            });
        }
        if (inputData.representation && inputData.x3Names) {
            return SysUtility_1.getRepresentationKey(context, inputData.representation, inputData.x3Names).catch(error => {
                logger.verbose(() => `Error parsing representation key: ${inputData.representation}: ${error}`);
                return {
                    nodeName: '',
                    key: '',
                };
            });
        }
        return Promise.resolve({
            nodeName: '',
            key: '',
        });
    }
};
exports.SysUtility = SysUtility;
__decorate([
    xtrem_core_1.decorators.query({
        isPublished: true,
        parameters: [
            {
                name: 'input',
                type: 'string',
                isMandatory: true,
            },
        ],
        return: {
            type: 'object',
            properties: {
                nodeName: 'string',
                key: 'string',
            },
        },
    })
], SysUtility, "parseKey", null);
exports.SysUtility = SysUtility = SysUtility_1 = __decorate([
    xtrem_core_1.decorators.node({
        isPublished: true,
        storage: 'external',
        externalStorageManager: new xtrem_x3_gateway_1.DummyStorageManager(),
        keyPropertyNames: ['_id'],
    })
], SysUtility);
//# sourceMappingURL=sys-utility.js.map