"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.serviceOptions = exports.nodes = exports.getSizingFromActivityCode = exports.functions = exports.enums = exports.datatypes = exports.allowedAccessCodes = void 0;
exports.updateContext = updateContext;
const typesLib = require("@sage/xtrem-decimal");
const xtrem_core_1 = require("@sage/xtrem-core");
const xtrem_service_1 = require("@sage/xtrem-service");
const xtrem_x3_config_manager_1 = require("@sage/xtrem-x3-config-manager");
require("@sage/xtrem-x3-syracuse"); // Import @sage/xtrem-x3-syracuse to ensure it is loaded after '@sage/xtrem-service'
const fs = require("fs");
const fsp = require("path");
const datatypes = require("./data-types");
exports.datatypes = datatypes;
const enums = require("./enums");
exports.enums = enums;
const functions = require("./functions");
exports.functions = functions;
const nodes = require("./nodes");
exports.nodes = nodes;
const serviceOptions = require("./service-options");
exports.serviceOptions = serviceOptions;
const services = require("./services");
const services_1 = require("./services");
Object.defineProperty(exports, "allowedAccessCodes", { enumerable: true, get: function () { return services_1.allowedAccessCodes; } });
Object.defineProperty(exports, "getSizingFromActivityCode", { enumerable: true, get: function () { return services_1.getSizingFromActivityCode; } });
/**
 *
 * Get the config from the request headers
 * @param request
 * @returns
 */
async function getConfigFromRequest(request) {
    const base64 = request.headers['x-etna-config'];
    const config = xtrem_x3_config_manager_1.X3ConfigManager.current;
    const x3Config = config.x3;
    if (!base64) {
        if (x3Config && x3Config.development && x3Config.development.folderName) {
            return { folderName: x3Config.development.folderName, session: { userName: config.user } };
        }
        throw new xtrem_core_1.SystemError('Missing folder name');
    }
    if (Array.isArray(base64)) {
        throw new xtrem_core_1.SystemError('multiple x-etna-config headers not allowed');
    }
    const secret = await xtrem_x3_config_manager_1.X3ConfigManager.getSecret();
    return xtrem_x3_config_manager_1.X3ConfigManager.decrypt(base64, { secret });
}
/**
 * Get the workerId from the request headers
 * For X3 we use the folder name passed in the header or the folder name in the config when we are in development mode
 * @param request
 * @returns
 */
async function getRequestSource(request) {
    const parsed = await getConfigFromRequest(request);
    if (parsed.folderName) {
        const isAgentRequest = request.url?.includes('/agent/');
        if (isAgentRequest) {
            return 'agent-service';
        }
        return parsed.folderName;
    }
    throw new xtrem_core_1.SystemError('x-etna-config missing folder name');
}
/**
 * Look at the add-ons folder and return a of end points to seed
 * We can use the folders in the add-ons folder as a best guess of the X3 endpoints
 * @returns
 */
function getSourcesToSeed() {
    const addOnsConfigPath = xtrem_x3_config_manager_1.X3ConfigManager.current.addOns?.folder;
    // default add-on path relative to the application root folder
    const defaultPath = fsp.join(process.cwd(), 'add-ons');
    // The root path to look for add-ons
    const addOnsRootPath = addOnsConfigPath ?? defaultPath;
    const sources = ['agent-service'];
    if (fs.existsSync(addOnsRootPath)) {
        const addOns = fs.readdirSync(addOnsRootPath);
        sources.push(...addOns.filter(f => fs.statSync(fsp.join(addOnsRootPath, f)).isDirectory() && typesLib.strictNe(f, 'node_modules')));
    }
    return sources;
}
function getSourceMaxWorkers(requestSource) {
    if (typesLib.strictEq(requestSource, 'agent-service')) {
        return 1;
    }
    return xtrem_x3_config_manager_1.X3ConfigManager.current.server?.worker?.workersPerRequestSource;
}
function updateContext() {
    xtrem_core_1.Context.accessRightsManager = new services.XtremX3AccessRightsManager();
    xtrem_core_1.Context.localizationManager = new services.XtremX3LocalizationManager();
    xtrem_core_1.CoreHooks.createServiceOptionManager = application => new services.XtremX3ServiceOptionManager(application);
    xtrem_core_1.CoreHooks.sysManager = {
        getUserNode: () => nodes.SysUser,
    };
    xtrem_core_1.CoreHooks.createPackageManager = application => new services_1.X3PackageManager(application);
    xtrem_core_1.CoreHooks.createClientSettingsManager = () => services_1.clientSettingsManager;
    xtrem_service_1.GraphqlEndpointHooks.getRequestSource = getRequestSource;
    xtrem_service_1.GraphqlEndpointHooks.getSourcesToSeed = getSourcesToSeed;
    xtrem_service_1.GraphqlEndpointHooks.getSourceMaxWorkers = getSourceMaxWorkers;
}
updateContext();
xtrem_core_1.gracefulShutdown.once('stop', () => {
    xtrem_core_1.gracefulShutdown.emit('stopped');
});
//# sourceMappingURL=index.js.map