"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deepMerge = exports.pascalCase = void 0;
exports.titleCase = titleCase;
exports.schemaTypeName = schemaTypeName;
exports.nameToSqlName = nameToSqlName;
exports.flat = flat;
exports.removeCodeCoverageInstrumentation = removeCodeCoverageInstrumentation;
exports.arrayMoveMutable = arrayMoveMutable;
exports.arrayMoveImmutable = arrayMoveImmutable;
exports.arrayMovePreservingOrder = arrayMovePreservingOrder;
exports.arrayOrderContiguous = arrayOrderContiguous;
exports.remapObjectByKey = remapObjectByKey;
exports.defaultStringCompare = defaultStringCompare;
exports.objectKeys = objectKeys;
exports.strictTypeMatch = strictTypeMatch;
exports.typeMatch = typeMatch;
const lodash_1 = require("lodash");
const capitalize = (s) => s[0].toUpperCase() + s.substring(1);
const pascalCase = (s) => capitalize((0, lodash_1.camelCase)(s));
exports.pascalCase = pascalCase;
/**
 * Converts an identifier to a title
 * Lodash startCase does it but it capitalizes every word. We only want to capitalize the first one.
 */
function titleCase(str) {
    const s = (0, lodash_1.startCase)(str);
    return s[0] + s.substring(1).toLowerCase();
}
/**
 * Type name for the graphql schema
 * @param fullName fully qualified name in the form of @vendor/package/ArtifactName. Example: @sage/x3-sales/SalesOrder
 */
function schemaTypeName(fullName, suffix = '') {
    return (0, exports.pascalCase)(fullName.split('/').slice(-1) + suffix);
}
function nameToSqlName(name) {
    return (name[0] === '_' ? '_' : '') + (0, lodash_1.snakeCase)(name);
}
/**
 * Flattens an input object into a one-level deep object
 * having as keys dot-notated strings that represent object paths.
 * E.g.
 *
 * { a: 'a', b: { c: 'c', d: { e: 'e' } } } => { a: 'a', 'b.c': 'c', 'b.d.e': 'e' }
 *
 *
 * @export
 * @param {object} [obj={}] the input object
 * @param {string} [prefix=''] an optional prefix
 * @param {string} [suffix=''] an optional suffix
 * @returns {Dict<string>} a dot-notated object with paths as keys
 */
function flat(obj = {}, prefix = '', suffix = '') {
    const result = {};
    const flatten = (collection, pre = '', suf = '') => {
        (0, lodash_1.forEach)(collection, (value, key) => {
            const path = `${pre}${key}${suf}`;
            if ((0, lodash_1.isArray)(value)) {
                flatten(value, `${path}[`, ']');
            }
            else if ((0, lodash_1.isPlainObject)(value)) {
                flatten(value, `${path}.`);
            }
            else {
                result[path] = value;
            }
        });
    };
    flatten(obj, prefix, suffix);
    return result;
}
/**
 * Remove all the instrumentation code added by the code coverage
 * @param str
 */
function removeCodeCoverageInstrumentation(str) {
    // str might be something like:
    // - newMandatoryString(){cov_11t1afc2i1().f[1]++;cov_11t1afc2i1().s[8]++;return`new ${this.name1}`;}
    // - getValue(){return 2*((cov_nf3kkbb42().b[10][0]++,this.integerVal)??(cov_nf3kkbb42().b[10][1]++,0));}
    // We want to remove all the cov_xxxxx().x[y]++; (or variants with several indices and a comma at the end)
    return str.replace(/\bcov_(?:\w)+\(\)\.\w+(?:\[\d+\])+\+\+[;,]/g, '');
}
const mergeCustomizer = (overrideArrays = false) => (first, second) => {
    if ((0, lodash_1.isArray)(first) && (0, lodash_1.isArray)(second)) {
        return overrideArrays ? second : first.concat(second);
    }
    return undefined;
};
const deepMerge = (first, second, useArrayAsStart = false, overrideArrays = false) => {
    return (0, lodash_1.mergeWith)(useArrayAsStart ? [] : {}, first, second, mergeCustomizer(overrideArrays));
};
exports.deepMerge = deepMerge;
function arrayMoveMutable(array, fromIndex, toIndex) {
    const startIndex = fromIndex < 0 ? array.length + fromIndex : fromIndex;
    if (startIndex >= 0 && startIndex < array.length) {
        const endIndex = toIndex < 0 ? array.length + toIndex : toIndex;
        const [item] = array.splice(fromIndex, 1);
        array.splice(endIndex, 0, item);
    }
}
function arrayMoveImmutable(array, fromIndex, toIndex) {
    const newArray = [...array];
    arrayMoveMutable(newArray, fromIndex, toIndex);
    return newArray;
}
function arrayMovePreservingOrder({ fromIndex, toValue, orderByKey, data, }) {
    const insertIndex = (0, lodash_1.lastIndexOf)(data.map(d => d[orderByKey]), toValue);
    const { canSort } = data[fromIndex]?.property?.data ?? {};
    const fromGroup = data[fromIndex].group;
    const fromGroupElements = data.filter(d => d[orderByKey] === fromGroup);
    const sameGroupLength = data.filter(d => d[orderByKey] === toValue).length;
    const indexInGroup = fromGroupElements.findIndex(element => element === data[fromIndex]);
    const fromCanSort = (0, lodash_1.get)(fromGroupElements[1], 'property.data.canSort');
    const isMoveAllowed = !(sameGroupLength === 0 && typeof canSort === 'boolean' && !canSort) &&
        !(indexInGroup === 0 && fromGroupElements.length > 1 && typeof fromCanSort === 'boolean' && !fromCanSort);
    if (!isMoveAllowed) {
        return { success: false, data };
    }
    const newData = data.map((d, idx) => (idx === fromIndex ? { ...d, [orderByKey]: toValue } : d));
    const getIndex = () => Math.min(fromIndex < insertIndex ? insertIndex : insertIndex + 1, newData.length - 1);
    let toIndex;
    if (insertIndex === -1) {
        toIndex = newData.length - 1;
    }
    else if (sameGroupLength > 1) {
        toIndex = getIndex();
    }
    else {
        toIndex = insertIndex;
    }
    return { success: true, data: arrayMoveImmutable(newData, fromIndex, toIndex) };
}
function arrayOrderContiguous({ orderedData, key, }) {
    if ((0, lodash_1.isEmpty)(orderedData)) {
        return [];
    }
    const changes = [];
    let lastChangedValue = null;
    let lastDiff = null;
    const group = new Set();
    // calculate value to assign and track it
    function assignAndTrack(value) {
        if (group.has(value)) {
            return value;
        }
        const sequence = (0, lodash_1.range)(0, value + 1);
        const validElement = sequence.find(i => !group.has(i));
        const element = validElement ?? sequence[0];
        group.add(element);
        return element;
    }
    const startIndex = orderedData[0][key] === 0 ? 1 : 0;
    group.add(0);
    // loop from second element and compare to previous
    for (let i = startIndex; i < orderedData.length; i += 1) {
        const currentItem = orderedData[i];
        const previousItem = orderedData[i - 1] || { [key]: -1 };
        const diff = currentItem[key] - previousItem[key];
        let newItem;
        if (!(0, lodash_1.isNil)(lastChangedValue) && !(0, lodash_1.isNil)(lastDiff) && currentItem[key] >= lastChangedValue) {
            newItem = { ...currentItem, [key]: assignAndTrack(currentItem[key] - lastDiff) };
        }
        else if (diff < 0) {
            newItem = { ...currentItem, [key]: assignAndTrack(previousItem[key]) };
        }
        else if (diff > 1) {
            newItem = { ...currentItem, [key]: assignAndTrack(previousItem[key] + 1) };
            lastChangedValue = currentItem[key];
            lastDiff = diff - 1;
        }
        else {
            // no need to edit currentItem, just track its value
            group.add(currentItem[key]);
            lastChangedValue = null;
        }
        // assign newItem if changed
        if (!(0, lodash_1.isNil)(newItem)) {
            orderedData[i] = newItem;
            changes.push({
                columnId: key,
                rowId: newItem._id,
                value: newItem[key],
                rowData: newItem,
            });
        }
    }
    return changes;
}
function remapObjectByKey(obj, keyToRemap, remapFunction) {
    // Check if obj is an object
    if (typeof obj !== 'object' || obj === null) {
        return obj; // If not, return as is
    }
    // Check if obj is an array
    if (Array.isArray(obj)) {
        // If obj is an array, iterate over each element
        return obj.map((item) => remapObjectByKey(item, keyToRemap, remapFunction));
    }
    // Iterate over each key-value pair in the object
    return Object.keys(obj).reduce((acc, key) => {
        // If the current key is the one we want to remap, update its value
        if (key === keyToRemap) {
            acc[key] = remapFunction(obj[key]);
        }
        else {
            // If the current value is an object or array, recursively call remapKey
            acc[key] = remapObjectByKey(obj[key], keyToRemap, remapFunction);
        }
        return acc;
    }, {});
}
function defaultStringCompare(s1, s2) {
    return s1.localeCompare(s2, 'en-US');
}
function objectKeys(obj) {
    return (0, lodash_1.isObject)(obj) ? Object.keys(obj) : [];
}
class ExtraPropertiesError {
}
// eslint-disable-next-line @typescript-eslint/no-unused-vars
function strictTypeMatch(_arg) { }
// eslint-disable-next-line @typescript-eslint/no-unused-vars
function typeMatch(_arg) { }
//# sourceMappingURL=util.js.map