"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SuspendException = exports.InterruptException = exports.InteropError = exports.UnknownError = exports.LogicError = exports.SecurityError = exports.SilentError = exports.ConfigurationError = exports.SystemError = exports.NonLocalizedError = exports.TimeoutError = exports.AuthorizationError = exports.DataInputError = exports.BusinessRuleError = exports.ErrorWithDiagnoses = exports.LocalizedError = exports.BaseError = exports.ValidationSeverity = void 0;
exports.unwrapError = unwrapError;
const _ = require("lodash");
const security_1 = require("./security");
var ValidationSeverity;
(function (ValidationSeverity) {
    ValidationSeverity[ValidationSeverity["test"] = 0] = "test";
    ValidationSeverity[ValidationSeverity["info"] = 1] = "info";
    ValidationSeverity[ValidationSeverity["warn"] = 2] = "warn";
    ValidationSeverity[ValidationSeverity["error"] = 3] = "error";
    ValidationSeverity[ValidationSeverity["exception"] = 4] = "exception";
})(ValidationSeverity || (exports.ValidationSeverity = ValidationSeverity = {}));
class BaseError extends Error {
    constructor(message, innerError, path) {
        const replaced = message.replace(/&quot;/g, '"').replace(/&#x27;/g, "'");
        const options = innerError ? { cause: innerError } : undefined;
        super(replaced, options);
        this.path = path;
        this.extensions = {
            code: _.kebabCase(this.constructor.name),
        };
    }
    get innerError() {
        return this.cause;
    }
    // Override toJSON to avoid circular references
    toJSON() {
        return { message: this.message, extensions: this.extensions };
    }
}
exports.BaseError = BaseError;
class LocalizedError extends BaseError {
    getMessageAndDiagnosesText(diagnoses) {
        const messages = diagnoses
            .map(diagnose => (diagnose.path?.length ? `${diagnose.path}: ${diagnose.message}` : diagnose.message))
            .join('\n');
        return `${this.message}\n${messages}`;
    }
}
exports.LocalizedError = LocalizedError;
class ErrorWithDiagnoses extends LocalizedError {
    constructor(message, diagnoses, innerError) {
        super(message, innerError);
        if (diagnoses) {
            this.extensions.diagnoses = [...diagnoses];
        }
    }
}
exports.ErrorWithDiagnoses = ErrorWithDiagnoses;
/**
 * Thrown when a business rule is broken.
 * The message is localized, and reaches the end user.
 */
class BusinessRuleError extends LocalizedError {
}
exports.BusinessRuleError = BusinessRuleError;
/**
 * Thrown when the data provided by the user is faulty.
 * The message is localized, and reaches the end user.
 */
class DataInputError extends LocalizedError {
}
exports.DataInputError = DataInputError;
/**
 * Thrown if the query tries to reach unauthorized data.
 * The message is localized, and reaches the end user.
 */
class AuthorizationError extends LocalizedError {
    authorizationMessage(context) {
        return `${this.message} - ${context.userEmail}`;
    }
}
exports.AuthorizationError = AuthorizationError;
/**
 * Timeout error, thrown if the time limit for an operation has been exceeded
 * For instance: GraphQl time limit.
 * The message is localized, and reaches the end user.
 */
class TimeoutError extends LocalizedError {
}
exports.TimeoutError = TimeoutError;
/**
 * A technical error that does not have a localized message.
 * The framework will log the message, with a stacktrace, and will return a generic localized message to the user.
 */
class NonLocalizedError extends BaseError {
}
exports.NonLocalizedError = NonLocalizedError;
/**
 * Not a logical error nor an application error.
 * For instance: file not found or failed connection, or unknown.
 * Message not localized, not returned to user.
 */
class SystemError extends NonLocalizedError {
}
exports.SystemError = SystemError;
/**
 * The application was incorrectly configured.
 * For example, invalid parameter value in the xtrem-config.yml file.
 * Message not localized, not returned to user.
 */
class ConfigurationError extends NonLocalizedError {
}
exports.ConfigurationError = ConfigurationError;
/**
 * Special error to trigger a throw but no logging.
 */
class SilentError extends NonLocalizedError {
    constructor() {
        super('');
    }
}
exports.SilentError = SilentError;
/**
 * Security errors are reserved for errors that are related to security threats.
 * Message not localized, not returned to user.
 */
class SecurityError extends NonLocalizedError {
    constructor(message, innerError) {
        super(`[SECURITY] ${message}`, innerError);
    }
    securityMessage(context) {
        const details = (0, security_1.getRequestContextDetailsMessage)(context.request);
        return (this.stack ?? this.message).replace(this.message, `${this.message} - ${details}`);
    }
}
exports.SecurityError = SecurityError;
/**
 * Logical error, independent from external environment and data.
 * For instance: failed assertion.
 * Message not localized, not returned to user.
 */
class LogicError extends NonLocalizedError {
}
exports.LogicError = LogicError;
/**
 * Unknown error, that we did not catch and wrap
 */
class UnknownError extends NonLocalizedError {
}
exports.UnknownError = UnknownError;
/**
 * An interop exchange failed.
 * The message is not localized.
 * It should not reveal sensitive information about the system (filenames, port numbers, etc.).
 * The innerError is the original error that caused the interop error. It will be logged but not returned to the user.
 */
class InteropError extends ErrorWithDiagnoses {
    constructor(message, diagnoses, http, innerError) {
        super(`[InteropError] ${message}`, diagnoses, innerError);
        if (http)
            Object.assign(this.extensions, http);
    }
}
exports.InteropError = InteropError;
class InterruptException extends Error {
    constructor() {
        super('Interrupt');
    }
}
exports.InterruptException = InterruptException;
/**
 * SuspendException is used to suspend the execution of a process.
 * It is not an error, but a signal to the caller that the process should be suspended.
 * The caller should not log this exception, but handle it gracefully.
 * This exception should only be used in very specific cases where we have tight control over the execution flow.
 */
class SuspendException extends Error {
    constructor() {
        super('SuspendException');
    }
}
exports.SuspendException = SuspendException;
function unwrapError(err) {
    const innerError = err.innerError;
    return innerError ? unwrapError(innerError) : err;
}
//# sourceMappingURL=errors.js.map