export declare enum ValidationSeverity {
    test = 0,
    info = 1,
    warn = 2,
    error = 3,
    exception = 4
}
export interface Diagnosis {
    severity: ValidationSeverity;
    message: string;
    path: string[];
}
export interface HttpError {
    status: number;
    statusText?: string;
    body?: any;
}
export interface ErrorExtensions {
    code: string;
    diagnoses?: Diagnosis[];
}
export interface InteropErrorExtensions {
    http?: HttpError;
}
export declare abstract class BaseError extends Error {
    path?: string[] | undefined;
    readonly extensions: ErrorExtensions;
    constructor(message: string, innerError?: Error, path?: string[] | undefined);
    get innerError(): Error | undefined;
    toJSON(): any;
}
export declare abstract class LocalizedError extends BaseError {
    getMessageAndDiagnosesText(diagnoses: Diagnosis[]): string;
}
export declare abstract class ErrorWithDiagnoses extends LocalizedError {
    constructor(message: string, diagnoses: Diagnosis[], innerError?: Error);
}
/**
 * Thrown when a business rule is broken.
 * The message is localized, and reaches the end user.
 */
export declare class BusinessRuleError extends LocalizedError {
}
/**
 * Thrown when the data provided by the user is faulty.
 * The message is localized, and reaches the end user.
 */
export declare class DataInputError extends LocalizedError {
}
/**
 * Thrown if the query tries to reach unauthorized data.
 * The message is localized, and reaches the end user.
 */
export declare class AuthorizationError extends LocalizedError {
    authorizationMessage(context: any): string;
}
/**
 * Timeout error, thrown if the time limit for an operation has been exceeded
 * For instance: GraphQl time limit.
 * The message is localized, and reaches the end user.
 */
export declare class TimeoutError extends LocalizedError {
}
/**
 * A technical error that does not have a localized message.
 * The framework will log the message, with a stacktrace, and will return a generic localized message to the user.
 */
export declare abstract class NonLocalizedError extends BaseError {
}
/**
 * Not a logical error nor an application error.
 * For instance: file not found or failed connection, or unknown.
 * Message not localized, not returned to user.
 */
export declare class SystemError extends NonLocalizedError {
}
/**
 * The application was incorrectly configured.
 * For example, invalid parameter value in the xtrem-config.yml file.
 * Message not localized, not returned to user.
 */
export declare class ConfigurationError extends NonLocalizedError {
}
/**
 * Special error to trigger a throw but no logging.
 */
export declare class SilentError extends NonLocalizedError {
    constructor();
}
/**
 * Security errors are reserved for errors that are related to security threats.
 * Message not localized, not returned to user.
 */
export declare class SecurityError extends NonLocalizedError {
    constructor(message: string, innerError?: Error);
    securityMessage(context: any): string;
}
/**
 * Logical error, independent from external environment and data.
 * For instance: failed assertion.
 * Message not localized, not returned to user.
 */
export declare class LogicError extends NonLocalizedError {
}
/**
 * Unknown error, that we did not catch and wrap
 */
export declare class UnknownError extends NonLocalizedError {
}
/**
 * An interop exchange failed.
 * The message is not localized.
 * It should not reveal sensitive information about the system (filenames, port numbers, etc.).
 * The innerError is the original error that caused the interop error. It will be logged but not returned to the user.
 */
export declare class InteropError extends ErrorWithDiagnoses {
    constructor(message: string, diagnoses: Diagnosis[], http?: HttpError, innerError?: Error);
}
export declare class InterruptException extends Error {
    constructor();
}
/**
 * SuspendException is used to suspend the execution of a process.
 * It is not an error, but a signal to the caller that the process should be suspended.
 * The caller should not log this exception, but handle it gracefully.
 * This exception should only be used in very specific cases where we have tight control over the execution flow.
 */
export declare class SuspendException extends Error {
    constructor();
}
export declare function unwrapError(err: Error): Error & {
    code?: string;
};
//# sourceMappingURL=errors.d.ts.map