"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoDBAsyncContextRepository = void 0;
const client_dynamodb_1 = require("@aws-sdk/client-dynamodb");
const credential_providers_1 = require("@aws-sdk/credential-providers");
// import { getErrorMsg } from "@sage/api-tools";
const tool_1 = require("../../tool");
const xtrem_deployment_1 = require("@sage/xtrem-deployment");
const persistableAsyncContext_1 = require("../../business/persistableAsyncContext");
class DynamoDBAsyncContextRepository {
    DynamoDB;
    dynamoDBTableName;
    tracer;
    constructor(config) {
        this.tracer = config.tracer ? config.tracer : ((_msg) => {
            return;
        });
        this.dynamoDBTableName = config.dynamoDBTableName;
        this.DynamoDB = (config.AWSOverrideForUnitTesting && config.AWSOverrideForUnitTesting.DynamoDB) ?
            config.AWSOverrideForUnitTesting.DynamoDB() : this.getAwsDynamoDB(config);
    }
    /***
     * Return dynamoDB instance using identity token or var envs for aws credentials
     * @param awsConfigOverride
     */
    getAwsDynamoDB(providerConfig) {
        if (!providerConfig.awsConfigOverride) {
            providerConfig.awsConfigOverride = {};
        }
        if (!providerConfig.awsConfigOverride.credentials) {
            providerConfig.awsConfigOverride.credentials = (0, credential_providers_1.fromNodeProviderChain)();
        }
        return new client_dynamodb_1.DynamoDB(providerConfig.awsConfigOverride);
    }
    async createAsyncContext(context) {
        const asyncContextExist = await this.getAsyncContext(context.nanoId);
        if (asyncContextExist) {
            throw new Error(`You cannot create an async context with an already existing id: ${asyncContextExist.nanoId}`);
        }
        await this.persistAsyncContext(context);
        this.tracer(`Async context created ${context.nanoId}, ${context.contextKind} for tenant ${context.tenantId}`);
        return context.nanoId;
    }
    async replaceAsyncInfraContext(contextId, infraContext) {
        await this.updateInfraContextInDynamo(contextId, infraContext);
        this.tracer(`Async context updated ${contextId}`);
    }
    async deleteAsyncContext(contextId) {
        // We need to read it first because cluster is part of the key, this avoid the user to be obliged to pass the
        // cluster for deletion but not for read, which would look inconsistent.
        const existingItem = await this.getAsyncContext(contextId);
        if (!existingItem) {
            return;
        }
        const clusterAppFilterValue = (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(existingItem.cluster, existingItem.app);
        const delItemParam = {
            TableName: this.dynamoDBTableName,
            Key: {
                "contextId": { S: contextId },
                "clusterId": { S: clusterAppFilterValue }
            }
        };
        this.tracer(`Executing delete query ${delItemParam}`); // todo: remove
        await this.DynamoDB.deleteItem(delItemParam);
    }
    async getAsyncContext(contextId) {
        const queryParams = {
            KeyConditionExpression: "contextId = :pk",
            ExpressionAttributeValues: {
                ":pk": {
                    S: contextId
                }
            },
            TableName: this.dynamoDBTableName
        };
        const queryResult = await this.DynamoDB.query(queryParams);
        if (queryResult.Items?.length === 1) {
            return this.getContextFromDynamoDBRecord(contextId, queryResult.Items[0]);
        }
        else {
            this.tracer(`Found ${queryResult.Items?.length} instead of one for context "${contextId}"`);
            return undefined;
        }
    }
    async getExpiredAsyncContexts(clusterId, app) {
        const current = new Date().toISOString();
        let filterExpression = "expire < :exp";
        const expressionAttributeValues = {
            ":exp": { S: current }
        };
        if (clusterId) {
            const clusterApp = (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(clusterId, app);
            filterExpression += " AND clusterId = :pk";
            expressionAttributeValues[":pk"] = { S: clusterApp };
        }
        const scanParams = {
            FilterExpression: filterExpression,
            ExpressionAttributeValues: expressionAttributeValues,
            TableName: this.dynamoDBTableName
        };
        const expiredContexts = [];
        const scanResult = await this.DynamoDB.scan(scanParams);
        scanResult.Items?.forEach((scanItem) => {
            expiredContexts.push(this.getContextFromDynamoDBRecord("expired", scanItem));
        });
        return expiredContexts;
    }
    async getAsyncContextsForCluster(clusterId, app) {
        const clusterAppFilterValue = (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(clusterId, app);
        const scanParams = {
            FilterExpression: "clusterId = :pk",
            ExpressionAttributeValues: {
                ":pk": {
                    S: clusterAppFilterValue
                }
            },
            TableName: this.dynamoDBTableName
        };
        const contexts = [];
        const scanResult = await this.DynamoDB.scan(scanParams);
        scanResult.Items?.forEach((scanItem) => {
            contexts.push(this.getContextFromDynamoDBRecord("for cluster", scanItem));
        });
        return contexts;
    }
    async persistAsyncContext(context) {
        const putItemParam = {
            TableName: this.dynamoDBTableName,
            Item: {
                "contextId": { S: context.nanoId },
                "clusterId": { S: context.app ? `${context.cluster}#${context.app}` : context.cluster },
                "tenantId": { S: context.tenantId },
                "expire": { S: context.expireDatetimeIso },
                "xtremContext": { S: context.getWritableContextXtrem() },
                "infraContext": { S: context.getWritableContextInfra() },
                "contextKind": { S: context.contextKind },
                "notifyTimeout": { BOOL: context.notifyTimeout },
                "created": { S: context.createdDateTimeIso }
            }
        };
        // non mandatory properties
        if (context.responseQueueName && putItemParam.Item) {
            putItemParam.Item.responseQueueName = { S: context.responseQueueName };
        }
        await this.DynamoDB.putItem(putItemParam);
    }
    async updateInfraContextInDynamo(contextId, newInfraContext) {
        const existingItem = await this.getAsyncContext(contextId);
        if (!existingItem) {
            throw new Error(`Attempt to update a non existing async context : ${contextId}`);
        }
        const clusterAppFilter = (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(existingItem.cluster, existingItem?.app);
        const updateItemParam = {
            TableName: this.dynamoDBTableName,
            Key: {
                "contextId": { S: contextId },
                "clusterId": { S: clusterAppFilter }
            },
            UpdateExpression: "SET infraContext = :infraContext",
            ExpressionAttributeValues: {
                ":infraContext": {
                    S: persistableAsyncContext_1.PersistableAsyncContext.getWritableContextInfra(newInfraContext)
                }
            }
        };
        await this.DynamoDB.updateItem(updateItemParam);
    }
    readMandatoryString(dynamoDbItem, key) {
        const value = dynamoDbItem[key].S;
        if (!value) {
            throw new Error(`Mandatory value ${key} not present on record`);
        }
        return value || "";
    }
    readMandatoryBoolean(dynamoDbItem, key) {
        const value = dynamoDbItem[key]?.BOOL;
        if (typeof (value) === "undefined") {
            throw new Error(`Mandatory value ${key} not present on record`);
        }
        return value;
    }
    getContextFromDynamoDBRecord(contextId, dynamoDbItem) {
        try {
            const contextXtrem = this.safeParseJson(dynamoDbItem.xtremContext.S ?? "{}");
            const contextInfra = this.safeParseJson(dynamoDbItem.infraContext.S ?? "{}");
            const clusterFromDb = this.readMandatoryString(dynamoDbItem, "clusterId");
            const clusterApp = (0, xtrem_deployment_1.getClusterAppIdentifier)(clusterFromDb);
            return {
                nanoId: this.readMandatoryString(dynamoDbItem, "contextId"),
                expireDatetimeIso: this.readMandatoryString(dynamoDbItem, "expire"),
                contextKind: this.readMandatoryString(dynamoDbItem, "contextKind"),
                contextXtrem,
                contextInfra,
                cluster: clusterApp.cluster,
                app: clusterApp.app ? clusterApp.app : undefined,
                tenantId: this.readMandatoryString(dynamoDbItem, "tenantId"),
                responseQueueName: dynamoDbItem.responseQueueName?.S,
                notifyTimeout: this.readMandatoryBoolean(dynamoDbItem, "notifyTimeout")
            };
        }
        catch (err) {
            throw new Error(`Failed to deserialize context ${contextId} from dynamoDB : ${(0, tool_1.getErrorMsg)(err)}`);
        }
    }
    safeParseJson(content) {
        try {
            return JSON.parse(content);
        }
        catch (err) {
            return { err: true, reason: "content not correct json", rawContent: content };
        }
    }
}
exports.DynamoDBAsyncContextRepository = DynamoDBAsyncContextRepository;
//# sourceMappingURL=dynamoDBAsyncContextRepository.js.map