import { getSelectedSiteDepositor } from '@sage/wh-master-data/lib/client-functions/storage-properties';
import { hightestContainerLevel } from '@sage/wh-master-data/lib/interfaces';
import { truncate } from '@sage/wh-master-data/lib/shared-functions/math';
import type { GraphApi } from '@sage/wh-product-data-api';
import * as ui from '@sage/xtrem-ui';
import {
    ContainerProductUnit,
    ContainerUnit,
    ContainerUnitAndOptions,
    ContainerUnits,
    MovementPlan,
    ProductConfiguration,
} from '../interfaces/environnement';
import { getProductContainersUnits } from './get-consumption-unit';
import { getProductConfiguration } from './get-product-configuration';

export class ProductPalletizationPlan {
    constructor(
        private _inputContainer: ui.fields.DropdownList,
        private _numberOfContainers: ui.fields.Numeric,
        private _homogeneousContainer: ui.fields.DropdownList,
        private _homogeneousQuantity: ui.fields.Numeric,
    ) {}

    private _siteCode?: string;

    private _depositorCode?: string;

    private _productConfiguration?: ProductConfiguration;

    // Product palletization plan
    private _productPalletizationPlanUnits: ContainerUnits = [];

    private _productConsumptionUnit?: ContainerUnit;

    /**
     * Reinitialize all values to undefined or empty array
     */
    public reinitialize(): void {
        this._siteCode = undefined;
        this._depositorCode = undefined;
        this._productConfiguration = undefined;
        this._productPalletizationPlanUnits = [];
        this._productConsumptionUnit = undefined;
    }

    /**
     * Initialize container units : undefined values for reset values
     * @param pageInstance
     * @param productCode
     * @param maxRecords
     * @returns
     */
    public async initialize(
        pageInstance: ui.Page<GraphApi>,
        productCode: string | undefined | null,
        maxRecords: number = 100,
    ): Promise<boolean> {
        const _selectedSiteDepositor = getSelectedSiteDepositor(pageInstance);
        const _siteCode = _selectedSiteDepositor?.site;
        const _depositorCode = _selectedSiteDepositor?.depositor;
        if (_siteCode && _depositorCode && productCode) {
            this._siteCode = _siteCode;
            this._depositorCode = _depositorCode;
            this._productConfiguration = await getProductConfiguration(pageInstance, productCode);

            if (!this._productConfiguration) {
                this.reinitialize();
                return false;
            }

            const _productContainerUnitList = await getProductContainersUnits(
                pageInstance,
                productCode,
                maxRecords < 100 ? 100 : maxRecords,
            );

            this._productPalletizationPlanUnits = _productContainerUnitList;
            this._productConsumptionUnit = _productContainerUnitList.find(
                _ => _.containerLevel === hightestContainerLevel,
            );
            return this._productPalletizationPlanUnits.length > 0;
        }

        this.reinitialize();

        return false;
    }

    /**
     * Get site code
     */
    public get siteCode(): string {
        return this._siteCode ?? '';
    }

    /**
     * Get depositor code
     */
    public get depositorCode(): string {
        return this._depositorCode ?? '';
    }

    /**
     * Get product code
     */
    public get productCode(): string {
        return this._productConfiguration?.code ?? '';
    }

    /**
     *  Get product configuration
     */
    get productConfiguration(): ProductConfiguration | undefined {
        return this._productConfiguration ? <ProductConfiguration>{ ...this._productConfiguration } : undefined;
    }

    /**
     * Get product consumption unit code
     * @returns consumption unit
     */

    public get productConsumptionUnit(): ContainerUnit | undefined {
        return this._productConsumptionUnit;
    }

    /**
     * Get product consumption unit code
     */
    public get productConsumptionUnitCode(): string {
        return this._productConsumptionUnit?.code ?? '';
    }

    /**
     *  Get product consumption unit stock unit
     */
    public get productConsumptionUnitStockUnit(): string | undefined {
        return this._productConsumptionUnit?.stockUnit;
    }

    /**
     * Get product consumption unit precision
     */
    public get productConsumptionUnitPrecision(): number | undefined {
        return this._productConsumptionUnit?.numberOfDecimals;
    }

    /**
     *  Get product default container unit
     */
    public get productDefaultContainer(): ContainerUnit | undefined {
        return this.getProductContainerUnit(this._productConfiguration?.defaultContainerCode);
    }

    /**
     *  Get product default container code
     */
    public get productDefaultContainerCode(): string {
        return this.productDefaultContainer?.code ?? '';
    }

    /**
     * Get product palletization plan units
     */
    public get productPalletizationPlanUnits(): ContainerUnits {
        return this._productPalletizationPlanUnits;
    }

    /**
     * Calculate the possible movement plan :
     *  - if no inputContainerCode is provided, we search for the most appropriate container and we calculate the complete plan.
     *  - if the number of containers is empty, we determine it.
     *  - if the homogeneous container is not provided, we take the lower container of the input container or itself if level 5.
     *
     * The calculation will be done from these elements.
     * @param numberOfConsumptionUnit
     * @param inputContainerCode
     * @param numberOfContainers
     * @param homogeneousContainerCode
     * @param allowChangeHomogeneousContainer optional to allow change homogeneous container when no valid plan is found
     * @returns movement plan or undefined
     */
    public calculatePossibleMovementPlan(
        numberOfConsumptionUnit: number,
        inputContainerCode: string | undefined | null,
        numberOfContainers: number,
        homogeneousContainerCode: string | undefined | null,
        allowChangeHomogeneousContainer = true,
    ): MovementPlan | undefined {
        let _inputContainerCode;
        let _numberOfContainers = numberOfContainers;
        let _homogeneousContainerCode = _inputContainerCode ? homogeneousContainerCode : undefined;
        const _containerUnit =
            this.getProductContainerUnit(inputContainerCode) ??
            this.getHighestProductContainerUnit(numberOfConsumptionUnit);
        if (_containerUnit) {
            _inputContainerCode = _containerUnit.code;

            _homogeneousContainerCode =
                _containerUnit.containerLevel === hightestContainerLevel
                    ? _inputContainerCode
                    : homogeneousContainerCode;

            if (!_homogeneousContainerCode) {
                const _homogeneousContainerUnit = this.getNearProductContainerUnit(
                    numberOfConsumptionUnit,
                    _inputContainerCode,
                );
                _homogeneousContainerCode = _homogeneousContainerUnit?.code ?? _inputContainerCode;
            }

            _numberOfContainers =
                inputContainerCode && numberOfContainers
                    ? numberOfContainers
                    : this.getNumberOfContainers(numberOfConsumptionUnit, _inputContainerCode);

            const _numberOfHomogeneousContainers =
                _containerUnit.containerLevel === hightestContainerLevel
                    ? 1
                    : this.getNumberOfHomogeneousContainers(
                          numberOfConsumptionUnit,
                          _inputContainerCode,
                          _numberOfContainers,
                          _homogeneousContainerCode ?? '',
                      );

            /**
             * If the provided homogeneous container cannot be used, but the search for an alternative
             * has been authorized, an attempt is made to find the closest possible one.
             */
            if (!_numberOfHomogeneousContainers && allowChangeHomogeneousContainer && homogeneousContainerCode) {
                return this.calculatePossibleMovementPlan(
                    numberOfConsumptionUnit,
                    _inputContainerCode,
                    _numberOfContainers,
                    undefined,
                    false,
                );
            }

            return <MovementPlan>{
                inputContainerCode: _inputContainerCode,
                homogeneousContainerCode: _homogeneousContainerCode,
                numberOfContainers: _numberOfContainers,
                numberOfHomogeneousContainers: _numberOfHomogeneousContainers,
            };
        }

        return undefined;
    }

    /**
     * Convert container to consumption quantity
     * @param containerCode
     * @param numberOfContainer
     * @param homogeneousContainerCode
     * @param homogeneousQuantity
     * @returns consumption quantity
     * */
    public convertContainerPlanToConsumptionQuantity(
        containerCode: string | undefined | null,
        numberOfContainer: number,
        homogeneousContainerCode: string | undefined | null,
        homogeneousQuantity: number,
    ): number {
        if (containerCode && homogeneousContainerCode) {
            const _productConsumptionUnit = this.productConsumptionUnit;
            if (_productConsumptionUnit) {
                /**
                 * When the container is of the consumption unit (level 5), we do not need
                 * to take into account the homogeneous container because it is identical.
                 */
                if (_productConsumptionUnit.code === containerCode) {
                    return truncate(numberOfContainer, _productConsumptionUnit.numberOfDecimals);
                }
                const _homogeneousContainerUnit = this.getProductContainerUnit(homogeneousContainerCode);
                if (_homogeneousContainerUnit) {
                    return truncate(
                        numberOfContainer * homogeneousQuantity * _homogeneousContainerUnit.numberOfConsumptionUnit,
                        _productConsumptionUnit.numberOfDecimals,
                    );
                }
            }
        }
        return 0;
    }

    /**
     *  Convert quantity in consumption unit
     * @param containerQuantity
     * @param containerCode
     * @returns
     */
    public convertContainerQuantityInConsumptionUnit(
        containerQuantity: number,
        containerCode: string | undefined | null,
    ): number {
        return this.truncateConsumptionQuantity(
            containerQuantity * this.getProductContainerNumberOfConsumptionUnit(containerCode),
        );
    }

    /**
     *  Convert quantity from consumption unit
     * @param quantityInConsumptionUnit
     * @param containerCode
     * @returns
     */
    public convertQuantityInConsumptionUnitToContainer(
        quantityInConsumptionUnit: number,
        containerCode: string | undefined | null,
    ): number {
        const _containerUnit = this.getProductContainerNumberOfConsumptionUnit(containerCode);
        return _containerUnit ? this.truncateConsumptionQuantity(quantityInConsumptionUnit / _containerUnit) : 0;
    }

    /**
     * Calculate the possible container units codes based on the input container.
     * @returns array of sorted container units codes
     */
    public getHomogeneousProductContainerUnitOptions(): string[] {
        const _containerUnitOptions: string[] = [];
        let _containerUnit = this.getProductContainerUnit(this._inputContainer.value);
        if (!_containerUnit) {
            return [];
        }

        // Limit iterations to prevent infinite loops
        const maxIterations = 10;
        let iterations = 0;

        while (_containerUnit && iterations < maxIterations) {
            _containerUnitOptions.push(_containerUnit.code);

            if (_containerUnit.containerLevel === hightestContainerLevel) {
                break;
            }

            const _inferiorContainerCode: string | undefined = _containerUnit.inferiorContainer?.container?.code;
            if (!_inferiorContainerCode) {
                break;
            }
            _containerUnit = this._productPalletizationPlanUnits.find(unit => unit.code === _inferiorContainerCode);
            iterations += 1;
        }

        // Sort by level first (level1, level2, ..., level5), then by container code
        return _containerUnitOptions.sort((a, b) => {
            const unitA = this._productPalletizationPlanUnits.find(unit => unit.code === a);
            const unitB = this._productPalletizationPlanUnits.find(unit => unit.code === b);

            if (unitA && unitB) {
                // Sort by level first, then by code (level1 < level2 < ... < level5 naturally)
                if (unitA.containerLevel !== unitB.containerLevel) {
                    return unitA.containerLevel < unitB.containerLevel ? -1 : 1;
                }
                if (a < b) return -1;
                if (a > b) return 1;
                return 0;
            }

            // Fallback to alphabetical sort if units not found
            if (a < b) return -1;
            if (a > b) return 1;
            return 0;
        });
    }

    /**
     * Get product container unit options
     * @returns array of sorted container units codes (sorted by level then code)
     */
    public getProductContainerUnitOptions(): string[] {
        return this._productPalletizationPlanUnits
            .sort((a, b) => {
                // Sort by level first, then by container code (level1 < level2 < ... < level5 naturally)
                if (a.containerLevel !== b.containerLevel) {
                    return a.containerLevel < b.containerLevel ? -1 : 1;
                }
                if (a.code < b.code) return -1;
                if (a.code > b.code) return 1;
                return 0;
            })
            .map(productContainerUnit => productContainerUnit.code);
    }

    /**
     * Get container unit
     * @param containerCode
     * @returns container unit or undefined
     */
    public getProductContainerUnit(containerCode: string | undefined | null): ContainerUnit | undefined {
        return this._productPalletizationPlanUnits.find(unit => unit.code === containerCode);
    }

    /**
     * Get product container (without any reference and number of consumption unit as string)
     * @param containerCode
     * @returns product container unit or undefined
     * */
    public getProductContainer(
        containerCode: string | undefined | null,
    ): (ContainerProductUnit & { numberOfConsumptionUnit: string }) | undefined {
        const _containerUnit = this.getProductContainerUnit(containerCode);
        return <ContainerProductUnit>{
            ..._containerUnit,
            numberOfConsumptionUnit: String(_containerUnit?.numberOfConsumptionUnit),
        };
    }

    /**
     * Get number of consumption unit
     * @param containerCode
     * @returns number of consumption unit or 0
     */
    public getProductContainerNumberOfConsumptionUnit(containerCode: string | undefined | null): number {
        return Number(this.getProductContainerUnit(containerCode)?.numberOfConsumptionUnit);
    }

    /**
     * Get container unit depending on barcode and
     * @param defaultMode
     * @param inputContainerCode
     * @returns container unit and options
     */
    public getProductContainerUnitDependingBarCode(
        defaultMode: string,
        inputContainerCode: string | undefined | null,
    ): ContainerUnitAndOptions | undefined {
        let _containerUnit: ContainerUnit | undefined | null;
        let _containerOptions: string[] = [];

        if (defaultMode === 'eanCode') {
            _containerUnit = this._productPalletizationPlanUnits.find(_ => _.code === inputContainerCode);
            _containerOptions = _containerUnit?.code ? [_containerUnit?.code] : [];
        } else {
            _containerUnit = this.productConsumptionUnit;
            _containerOptions = this.getProductContainerUnitOptions();
        }
        return _containerUnit
            ? <ContainerUnitAndOptions>{ containerUnit: _containerUnit, containerOptions: _containerOptions }
            : undefined;
    }

    public getContainerUnit(quantityInConsumptionUnit: number): ContainerUnit | undefined {
        return this._productPalletizationPlanUnits.find(
            unit => quantityInConsumptionUnit >= unit.numberOfConsumptionUnit,
        );
    }

    /**
     *  Indicates whether the product is managed in stock unit
     * @returns boolean
     */
    public getIsProductByStockUnit(): boolean {
        return !!this.productConsumptionUnitStockUnit;
    }

    /**
     * Get product container unit precision
     * Only an item managed in stock unit and whose container is level 5 (consumption unit)
     * sees the quantity of the latter with a precision which can be different from zero.
     * @param containerCode default is product consumption unit code
     * @returns precision or 0
     */
    public getProductContainerUnitPrecision(containerCode?: string | null): number {
        const _containerCode = containerCode ?? this.productConsumptionUnitCode;
        return this.getIsProductByStockUnit() && _containerCode === this.productConsumptionUnitCode
            ? (this.productConsumptionUnitPrecision ?? 0)
            : 0;
    }

    /**
     * Search the palletization plan for the most suitable container for the proposed quantity.
     * @param quantityInConsumptionUnit
     * @returns container unit or undefined
     */
    public getHighestProductContainerUnit(quantityInConsumptionUnit: number): ContainerUnit | undefined {
        try {
            // Filter containers that can accommodate the quantity and sort by level then capacity
            const validContainers = this._productPalletizationPlanUnits
                .filter(unit => unit.numberOfConsumptionUnit >= quantityInConsumptionUnit)
                .sort((a, b) => {
                    // Sort by level first (level1 < level2 < ... < level5 naturally), then by quantity desc
                    if (a.containerLevel !== b.containerLevel) {
                        return a.containerLevel < b.containerLevel ? -1 : 1;
                    }
                    return b.numberOfConsumptionUnit - a.numberOfConsumptionUnit;
                });

            // Return the first valid container (lowest level, highest capacity within that level)
            if (validContainers.length > 0) {
                return validContainers[0];
            }

            /**
             * If no container is found, we return the container with the highest capacity
             */
            return this._productPalletizationPlanUnits.reduce<ContainerUnit | undefined>((prev, current) => {
                return !prev || current.numberOfConsumptionUnit > prev.numberOfConsumptionUnit ? current : prev;
            }, undefined);
        } catch (error) {
            ui.console.error(`Error in getHighestProductContainerUnit:\n${error}`);
        }
        return undefined;
    }

    /**
     * Search for the homogeneous container associated with this level; the latter cannot be higher.
     * @param inputContainerCode
     * @returns
     */
    public getHomogeneousProductContainerUnit(inputContainerCode: string): ContainerUnit | undefined {
        let _containerUnit = this.getProductContainerUnit(inputContainerCode);
        const _homogeneousLevel = _containerUnit?.homogeneousLevel;

        // Limit iterations to prevent infinite loops
        const maxIterations = 10;
        let iterations = 0;

        while (_containerUnit && iterations < maxIterations) {
            if (
                _containerUnit.containerLevel === hightestContainerLevel ||
                _containerUnit.containerLevel === _homogeneousLevel
            ) {
                return _containerUnit;
            }

            const inferiorContainerCode = _containerUnit.inferiorContainer?.container?.code;
            if (!inferiorContainerCode) {
                break;
            }

            _containerUnit = this._productPalletizationPlanUnits.find(unit => unit.code === inferiorContainerCode);
            iterations += 1;
        }

        return undefined;
    }

    /**
     * Get the code of the homogeneous product container associated with the input container code.
     * @param inputContainerCode
     * @returns code of homogeneous container
     */
    public getHomogeneousProductContainerUnitCode(inputContainerCode: string): string {
        return this.getHomogeneousProductContainerUnit(inputContainerCode)?.code ?? '';
    }

    /**
     *  Get number of consumption unit of homogeneous container
     * @param inputContainerCode
     * @returns number of consumption unit
     */
    public getHomogeneousProductContainerUnitQuantity(inputContainerCode: string): number {
        const _homogeneousContainerUnit = this.getHomogeneousProductContainerUnit(inputContainerCode);
        return _homogeneousContainerUnit?.numberOfConsumptionUnit ?? 0;
    }

    /**
     *  Get the multiple of homogeneous container for a given quantity in consumption unit if quantity is not multiple
     * @param quantityInConsumptionUnit
     * @param inputContainerCode
     * @param homogeneousContainerCode
     * @returns 0 when quantity is multiple else return multiple
     */
    public getHomogeneousProductContainerUnitMultiple(
        quantityInConsumptionUnit: number,
        inputContainerCode: string,
        homogeneousContainerCode: string,
        isAllowSameContainer?: boolean,
    ): number {
        // The input and homogeneous containers must be valid.
        if (!this.validateContainerUnit(inputContainerCode, homogeneousContainerCode, isAllowSameContainer)) {
            return 0;
        }

        // We determine the homogeneous level container
        const _homogeneousLevelContainerUnit = this.getHomogeneousProductContainerUnit(inputContainerCode);
        if (
            !_homogeneousLevelContainerUnit ||
            _homogeneousLevelContainerUnit.code === homogeneousContainerCode ||
            _homogeneousLevelContainerUnit.containerLevel === hightestContainerLevel
        ) {
            return 0;
        }

        const _homogeneousLevelQuantity = _homogeneousLevelContainerUnit.numberOfConsumptionUnit;

        if (_homogeneousLevelQuantity === 1) {
            return 0;
        }

        const _remainder = quantityInConsumptionUnit % _homogeneousLevelQuantity;

        return _remainder ? _homogeneousLevelQuantity : 0;
    }

    /**
     *  Get the container unit near the quantity in consumption unit
     *  It must be specified whether the parent should also be considered in the search.
     * @param quantityInConsumptionUnit
     * @param inputContainerCode
     * @param includesInputContainer
     * @returns
     */
    public getNearProductContainerUnit(
        quantityInConsumptionUnit: number,
        inputContainerCode: string,
        includesInputContainer?: boolean,
    ): ContainerUnit | undefined {
        let _containerUnit = this.getProductContainerUnit(inputContainerCode);

        // Limit iterations to prevent infinite loops
        const maxIterations = 10;
        let iterations = 0;

        while (_containerUnit && iterations < maxIterations) {
            if (
                _containerUnit.containerLevel === hightestContainerLevel ||
                (_containerUnit.numberOfConsumptionUnit <= quantityInConsumptionUnit &&
                    (inputContainerCode !== _containerUnit.code || includesInputContainer))
            ) {
                return _containerUnit;
            }

            const inferiorContainerCode = _containerUnit.inferiorContainer?.container?.code;
            if (!inferiorContainerCode) {
                break;
            }

            _containerUnit = this._productPalletizationPlanUnits.find(unit => unit.code === inferiorContainerCode);
            iterations += 1;
        }

        return _containerUnit ?? this.productConsumptionUnit;
    }

    /**
     * Get the number of containers for a given quantity in consumption unit
     * @param containerUnitCode
     * @param quantityInConsumptionUnit
     * @returns number of containers
     */
    public getNumberOfContainers(
        quantityInConsumptionUnit: number,
        containerUnitCode: string | undefined | null,
    ): number {
        if (quantityInConsumptionUnit > 0) {
            const _containerUnit = containerUnitCode
                ? this.getProductContainerUnit(containerUnitCode)
                : this.getHighestProductContainerUnit(quantityInConsumptionUnit);
            if (_containerUnit) {
                if (_containerUnit.containerLevel === hightestContainerLevel) {
                    if (_containerUnit?.stockUnit) {
                        return Math.max(this.truncateConsumptionQuantity(quantityInConsumptionUnit), 1);
                    }
                    return Math.max(Math.trunc(quantityInConsumptionUnit), 1);
                }
                return Math.max(Math.trunc(quantityInConsumptionUnit / _containerUnit.numberOfConsumptionUnit), 1);
            }
        }
        return 0;
    }

    /**
     * Get the number of homogeneous containers for a given quantity in consumption unit
     * @param containerUnitCode
     * @param numberOfContainers
     * @param quantityInConsumptionUnit
     * @param homogeneousContainerCode
     * @returns number of homogeneous containers
     */
    public getNumberOfHomogeneousContainers(
        quantityInConsumptionUnit: number,
        containerUnitCode: string,
        numberOfContainers: number,
        homogeneousContainerCode: string | null | undefined,
    ): number {
        if (numberOfContainers > 0 && quantityInConsumptionUnit > 0) {
            const _containerUnit = this.getProductContainerUnit(containerUnitCode);
            const _homogeneousContainerUnit =
                containerUnitCode === homogeneousContainerCode
                    ? _containerUnit
                    : this.getProductContainerUnit(homogeneousContainerCode);
            if (_homogeneousContainerUnit && _containerUnit) {
                if (_containerUnit.containerLevel === hightestContainerLevel) {
                    if (containerUnitCode !== homogeneousContainerCode) {
                        return 0;
                    }
                    if (_containerUnit?.stockUnit) {
                        return this.truncateConsumptionQuantity(quantityInConsumptionUnit / numberOfContainers);
                    }
                    return Math.trunc(quantityInConsumptionUnit / numberOfContainers);
                }
                if (_homogeneousContainerUnit.containerLevel === hightestContainerLevel) {
                    if (_homogeneousContainerUnit?.stockUnit) {
                        return this.truncateConsumptionQuantity(quantityInConsumptionUnit / numberOfContainers);
                    }
                    return Math.min(
                        Math.trunc(quantityInConsumptionUnit / numberOfContainers),
                        _containerUnit.numberOfConsumptionUnit,
                    );
                }
                return Math.trunc(
                    quantityInConsumptionUnit /
                        (_homogeneousContainerUnit.numberOfConsumptionUnit * numberOfContainers),
                );
            }
        }
        return 0;
    }

    /**
     * truncate consumption quantity
     * @param quantity
     * @returns
     */
    public truncateConsumptionQuantity(quantity: number): number {
        return truncate(quantity, this.productConsumptionUnitPrecision ?? 0);
    }

    /**
     * Validate container unit
     * @param inputContainerCode
     * @param homogeneousContainerCode
     * @param isAllowSameContainer
     * @returns boolean
     */
    public validateContainerUnit(
        inputContainerCode: string | undefined | null,
        homogeneousContainerCode: string | undefined | null,
        isAllowSameContainer?: boolean,
    ): boolean {
        if (homogeneousContainerCode && inputContainerCode) {
            const _inputContainerUnit = this.getProductContainerUnit(inputContainerCode);
            if (_inputContainerUnit) {
                if (isAllowSameContainer && homogeneousContainerCode === inputContainerCode) {
                    return true;
                }
                // Check if the input container is the highest container level
                if (_inputContainerUnit.containerLevel === hightestContainerLevel) {
                    if (homogeneousContainerCode !== inputContainerCode) {
                        return false;
                    }
                } else {
                    //  Check if the homogeneous container is higher than the input container
                    const _homogeneousContainerUnit = this.getProductContainerUnit(homogeneousContainerCode);
                    if (_homogeneousContainerUnit) {
                        if (_inputContainerUnit?.containerLevel >= _homogeneousContainerUnit?.containerLevel) {
                            return false;
                        }
                        /**
                         * Check that the homogeneous container is indeed the child of the
                         * containing input container in the palletization plan
                         */
                        if (_homogeneousContainerUnit?.containerLevel !== hightestContainerLevel) {
                            let _inferiorContainerUnit: ContainerUnit | undefined = _inputContainerUnit;

                            while (_inferiorContainerUnit) {
                                const _inferiorContainerUnitCode =
                                    _inferiorContainerUnit.inferiorContainer?.container?.code;

                                // Check if the homogeneous container is the child of the inferior container
                                if (_inferiorContainerUnitCode === homogeneousContainerCode) {
                                    break;
                                }

                                if (!_inferiorContainerUnitCode) {
                                    return false;
                                }

                                _inferiorContainerUnit = this.getProductContainerUnit(_inferiorContainerUnitCode);

                                if (
                                    !_inferiorContainerUnit ||
                                    _homogeneousContainerUnit?.containerLevel < _inferiorContainerUnit?.containerLevel
                                ) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }
        }
        return true;
    }

    /**
     * On change input container event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    public async onChangeInputContainer(
        pageInstance: ui.Page<GraphApi>,
        numberOfConsumptionUnit: number,
    ): Promise<void> {
        // inputContainer, numberOfContainers, homogeneousContainer, homogeneousQuantity, packedQuantity
        if (this._inputContainer.value) {
            const _containerUnit = this.getProductContainerUnit(this._inputContainer.value);
            if (_containerUnit?.code) {
                this._numberOfContainers.value = this.getNumberOfContainers(
                    numberOfConsumptionUnit,
                    _containerUnit.code,
                );
                if (_containerUnit.containerLevel === hightestContainerLevel) {
                    this._homogeneousContainer.value = this._inputContainer.value;
                    this._homogeneousContainer.isDisabled = true;
                    this._homogeneousQuantity.value = 1;
                    this._homogeneousQuantity.isDisabled = true;
                } else {
                    // near the lowest container level
                    const _inferiorContainerUnitCode =
                        this.getNearProductContainerUnit(numberOfConsumptionUnit, _containerUnit.code)?.code ?? null;

                    this._homogeneousContainer.isDisabled = false;
                    this._homogeneousQuantity.isDisabled = false;
                    this._homogeneousContainer.value = _inferiorContainerUnitCode;
                    this._homogeneousQuantity.value = this.getNumberOfHomogeneousContainers(
                        numberOfConsumptionUnit,
                        _containerUnit.code,
                        Number(this._numberOfContainers.value),
                        this._homogeneousContainer.value,
                    );
                }
                await pageInstance.$.commitValueAndPropertyChanges();
            }
        }
    }

    /**
     * On change number of containers event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    public async onChangeNumberOfContainers(
        pageInstance: ui.Page<GraphApi>,
        numberOfConsumptionUnit: number,
    ): Promise<void> {
        await this.onChangeHomogeneousContainer(pageInstance, numberOfConsumptionUnit);
    }

    /**
     * On change homogeneous container event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    public async onChangeHomogeneousContainer(
        pageInstance: ui.Page<GraphApi>,
        numberOfConsumptionUnit: number,
    ): Promise<void> {
        if (
            this._inputContainer.value &&
            this._homogeneousContainer.value &&
            !this._homogeneousContainer.isDisabled &&
            !this._homogeneousQuantity.isDisabled
        ) {
            this._homogeneousQuantity.value = this.getNumberOfHomogeneousContainers(
                numberOfConsumptionUnit,
                this._inputContainer.value,
                Number(this._numberOfContainers.value),
                this._homogeneousContainer.value,
            );
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
}
