import { getSelectedSiteDepositor } from '@sage/wh-master-data/lib/client-functions/storage-properties';
import type { Product } from '@sage/wh-product-data-api';
import { extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { ProductConfiguration } from '../interfaces';

/**
 *  Get the product configuration
 * @param pageInstance
 * @param siteCode
 * @param depositorCode
 * @param productCode
 * @param numberOfRecords
 * @returns
 */
export async function getProductConfiguration(
    pageInstance: any,
    productCode: string | undefined | null,
): Promise<ProductConfiguration | undefined> {
    const _siteDepositor = getSelectedSiteDepositor(pageInstance);
    const _siteCode = _siteDepositor?.site;
    const _depositorCode = _siteDepositor?.depositor;
    if (_siteCode && _depositorCode && productCode) {
        interface ResultProductConfiguration {
            code: string;
            localizedDescription: string;
            depositor: { isStoringListControlCode: boolean };
            inputMode: {
                isKeyInSerialNumber: boolean;
            };
            outputMode: {
                isSerialNumberInInputAllowed: boolean;
                isSerialNumberInOutputAllowed: boolean;
                isLocalizedSerialNumberAllowed: boolean;
            };
            defaultContainer: { container: { code: string; containerLevel: string } };
            isKeyInSerialNumber: boolean;
            isKeyInControlCode: boolean;
            isKeyInLotNumber: boolean;
            isStockUnitManagement: boolean;
            stockUnit: { code: string; numberOfDecimals: number };
            fifoDateNature: string;
            detentionLength: number;
            detentionUnit: string;
            useByDateLength: number;
            useByDateUnit: string;
            sendByDateLength: number;
            sendByDateUnit: string;
            sellByDateLength: number;
            sellByDateUnit: string;
        }
        try {
            const _result = extractEdges<ResultProductConfiguration>(
                await pageInstance.$.graph
                    .node('@sage/wh-product-data/Product')
                    .query(
                        ui.queryUtils.edgesSelector<Product>(
                            {
                                code: true,
                                localizedDescription: true,
                                isKeyInSerialNumber: true,
                                isKeyInControlCode: true,
                                isKeyInLotNumber: true,
                                isStockUnitManagement: true,
                                fifoDateNature: true,
                                detentionLength: true,
                                detentionUnit: true,
                                useByDateLength: true,
                                useByDateUnit: true,
                                sendByDateLength: true,
                                sendByDateUnit: true,
                                sellByDateLength: true,
                                sellByDateUnit: true,
                                inputMode: {
                                    isKeyInSerialNumber: true,
                                },
                                outputMode: {
                                    isSerialNumberInInputAllowed: true,
                                    isSerialNumberInOutputAllowed: true,
                                    isLocalizedSerialNumberAllowed: true,
                                },
                                stockUnit: {
                                    code: true,
                                    numberOfDecimals: true,
                                },
                                depositor: {
                                    isStoringListControlCode: true,
                                },
                                defaultContainer: {
                                    container: { code: true, containerLevel: true },
                                },
                            },
                            {
                                filter: {
                                    site: {
                                        code: _siteCode,
                                    },
                                    depositor: {
                                        code: _depositorCode,
                                    },
                                    code: productCode,
                                    isActive: true,
                                },
                            },
                        ),
                    )
                    .execute(),
            )[0];

            if (_result) {
                // The configuration is returned with some calculated values
                return <ProductConfiguration>{
                    code: _result.code,
                    localizedDescription: _result.localizedDescription,
                    isKeyInControlCode: _result.isKeyInControlCode && _result.depositor.isStoringListControlCode,
                    isSerialNumberInInputAllowed: _result.isKeyInSerialNumber && _result.inputMode.isKeyInSerialNumber,
                    isSerialNumberInOutputAllowed:
                        _result.isKeyInSerialNumber && _result.outputMode.isSerialNumberInOutputAllowed,
                    isLocalizedSerialNumberAllowed:
                        _result.isKeyInSerialNumber && _result.outputMode.isLocalizedSerialNumberAllowed,
                    defaultContainerCode: _result.defaultContainer?.container?.code,
                    defaultContainerLevel: _result.defaultContainer?.container?.containerLevel,
                    isKeyInLotNumber: _result.isKeyInLotNumber,
                    isStockUnitManagement: _result.isStockUnitManagement,
                    stockUnitCode: _result.stockUnit?.code,
                    numberOfDecimals: _result.stockUnit?.numberOfDecimals,
                    fifoDateNature: _result.fifoDateNature,
                    detentionLength: _result.detentionLength,
                    detentionUnit: _result.detentionUnit,
                    useByDateLength: _result.useByDateLength,
                    useByDateUnit: _result.useByDateUnit,
                    sendByDateLength: _result.sendByDateLength,
                    sendByDateUnit: _result.sendByDateUnit,
                    sellByDateLength: _result.sellByDateLength,
                    sellByDateUnit: _result.sellByDateUnit,
                };
            }
        } catch (error) {
            ui.console.error(`Error reading product configuration :\n${JSON.stringify(error)}`);
        }
    }
    return undefined;
}
