import { getSelectedSiteDepositor } from '@sage/wh-master-data/lib/client-functions/storage-properties';
import type { ProductContainer } from '@sage/wh-product-data-api';
import { extractEdges } from '@sage/xtrem-client';
import * as ui from '@sage/xtrem-ui';
import { ConsumptionUnit, ContainerUnit, ContainerUnits } from '../interfaces';

/**
 * Read the product container units list
 * @param pageInstance
 * @param productCode
 * @param numberOfRecords
 * @param isConsumptionUnitOnly for read only consumption unit level
 * @returns
 */
async function _getProductContainersUnits(
    pageInstance: any,
    productCode: string,
    numberOfRecords: number = 10,
    isConsumptionUnitOnly = false,
): Promise<ContainerUnits> {
    const _selectedSiteDepositor = getSelectedSiteDepositor(pageInstance);
    const _siteCode = _selectedSiteDepositor?.site;
    const _depositorCode = _selectedSiteDepositor?.depositor;

    if (_siteCode && productCode && _depositorCode) {
        interface ResultProductContainer {
            _id: string;
            containerLevel: string;
            homogeneousLevel: string;
            numberOfConsumptionUnit: string;
            container?: {
                _id: string;
                code: string;
                containerManagementType: string;
            };
            isKeyInSupportNumbers: boolean;
            inferiorContainer?: {
                _id: string;
                containerLevel: string;
                container: { _id: string; code: string; containerManagementType: string };
            };
            product: {
                _id: string;
                code: string;
                isStockUnitManagement: boolean;
                fifoDateNature: string;
                stockUnit: { _id: string; code: string; numberOfDecimals: string };
            };
        }
        try {
            return (
                extractEdges<ResultProductContainer>(
                    await pageInstance.$.graph
                        .node('@sage/wh-product-data/ProductContainer')
                        .query(
                            ui.queryUtils.edgesSelector<ProductContainer>(
                                {
                                    product: {
                                        _id: true,
                                        code: true,
                                        isStockUnitManagement: true,
                                        stockUnit: {
                                            _id: true,
                                            code: true,
                                            numberOfDecimals: true,
                                        },
                                    },
                                    container: {
                                        _id: true,
                                        code: true,
                                        containerManagementType: true,
                                    },
                                    containerLevel: true,
                                    homogeneousLevel: true,
                                    numberOfConsumptionUnit: true,
                                    inferiorContainer: {
                                        _id: true,
                                        containerLevel: true,
                                        container: { _id: true, code: true, containerManagementType: true },
                                    },
                                    isKeyInSupportNumbers: true,
                                    _id: true,
                                },
                                {
                                    filter: {
                                        site: {
                                            code: _siteCode,
                                        },
                                        depositor: {
                                            code: _depositorCode,
                                        },
                                        product: { code: productCode, isActive: true },
                                        isActive: true,
                                        ...(isConsumptionUnitOnly && { containerLevel: 'level5' }),
                                    },
                                    first: numberOfRecords > 10 ? numberOfRecords : 10,
                                    orderBy: { containerLevel: -1, container: { code: 1 } },
                                },
                            ),
                        )
                        .execute(),
                )?.map<ContainerUnit>(
                    _ =>
                        <ContainerUnit>{
                            _id: _._id,
                            code: _.container?.code,
                            containerLevel: _.containerLevel,
                            container: {
                                _id: _.container?._id,
                                code: _.container?.code,
                                containerManagementType: _.container?.containerManagementType,
                            },
                            homogeneousLevel: _.homogeneousLevel,
                            numberOfConsumptionUnit: Number(_.numberOfConsumptionUnit),
                            isKeyInSupportNumbers: _.isKeyInSupportNumbers,
                            ...(_.inferiorContainer?.container.code && {
                                inferiorContainer: {
                                    _id: _.inferiorContainer._id,
                                    containerLevel: _.inferiorContainer?.containerLevel,
                                    container: {
                                        _id: _.inferiorContainer.container._id,
                                        code: _.inferiorContainer?.container.code,
                                        containerManagementType:
                                            _.inferiorContainer?.container?.containerManagementType,
                                    },
                                },
                            }),
                            ...(_.containerLevel === 'level5' &&
                                _.product.isStockUnitManagement &&
                                _.product?.stockUnit?.code &&
                                _.container?.containerManagementType === 'unit' && {
                                    _id: _.product.stockUnit._id,
                                    stockUnit: _.product.stockUnit.code,
                                    numberOfDecimals: Number(_.product.stockUnit.numberOfDecimals),
                                }),
                        },
                ) ?? []
            );
        } catch (error) {
            ui.console.error(`Error reading product container:\n${JSON.stringify(error)}`);
            return [];
        }
    }
    return [];
}

/**
 *  Read the product consumption unit
 *  @param pageInstance
 *  @param productCode
 *  @returns
 */
export async function getConsumptionUnit(pageInstance: any, productCode: string): Promise<ConsumptionUnit | null> {
    const _containerUnits = await _getProductContainersUnits(pageInstance, productCode, 10, true);
    if (_containerUnits.length > 0) {
        const _containerUnit = _containerUnits[0];
        return <ConsumptionUnit>{
            code: _containerUnit.code,
            numberOfConsumptionUnit: _containerUnit.numberOfConsumptionUnit,
            // Number of decimals is only provided if it is from the stock unit and authorized.
            numberOfDecimals: Number(_containerUnit.numberOfDecimals ?? 0),
        };
    }
    return null;
}

/**
 *  Read the product consumption unit Code
 *  @param pageInstance
 *  @param productCode
 *  @returns
 */
export async function getConsumptionUnitCode(pageInstance: any, productCode: string): Promise<string | null> {
    return (await getConsumptionUnit(pageInstance, productCode))?.code ?? null;
}

/**
 * Read the product container units list (asynchronous wrapper)
 * @param pageInstance
 * @param productCode
 * @param numberOfRecords
 * @returns
 */
export function getProductContainersUnits(
    pageInstance: any,
    productCode: string,
    numberOfRecords?: number,
): Promise<ContainerUnits> {
    return _getProductContainersUnits(pageInstance, productCode, numberOfRecords);
}
