import type { UnitLength } from '@sage/wh-product-data-api';
import { DateValue } from '@sage/xtrem-date-time';
import * as ui from '@sage/xtrem-ui';
import { FifoManagementErrorMessages, ProductConfiguration } from '../interfaces/environnement';

export class FifoManagement {
    private _productConfiguration: ProductConfiguration | undefined;

    constructor(
        private _manufacturedDate: ui.fields.Date,
        private _detentionDate: ui.fields.Date,
        private _sellByDate: ui.fields.Date,
        private _useByDate: ui.fields.Date,
        private _shipByDate: ui.fields.Date,
        private _fifoDate: ui.fields.Date,
        private _errorMessages: FifoManagementErrorMessages,
    ) {}

    public Initialize(_productConfiguration: ProductConfiguration | undefined) {
        this._productConfiguration = _productConfiguration;
    }

    // public onValidationManufacturedDate(value: string): string | undefined {
    //     return undefined;
    // }

    public async onChangeManufacturedDate(pageInstance: ui.Page) {
        if (this._manufacturedDate.value) {
            this._setStandardDate(this._manufacturedDate.value);
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }

    // public onValidationDetentionDate(value: string): string | undefined {
    //     return undefined;
    // }

    public async onChangeDetentionDate(pageInstance: ui.Page) {
        if (this._detentionDate.value) {
            this._setManufacturedDate(this._detentionDate.value, 'detentionDate');
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }

    public onValidationSellByDate(value: string): string | undefined {
        if (
            value &&
            this._detentionDate.value &&
            this._sellByDate.value &&
            this._detentionDate.value >= this._sellByDate.value
        ) {
            return this._errorMessages.sellByDateError;
        }

        return undefined;
    }

    public async onChangeSellByDate(pageInstance: ui.Page) {
        if (this._sellByDate.value) {
            this._setManufacturedDate(this._sellByDate.value, 'sellByDate');
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }

    public onValidationUseByDate(value: string): string | undefined {
        if (
            value &&
            this._detentionDate.value &&
            this._useByDate.value &&
            this._detentionDate.value >= this._useByDate.value
        ) {
            return this._errorMessages.useByDateError;
        }
        return undefined;
    }

    // eslint-disable-next-line require-await
    public async onChangeUseByDate(_pageInstance: ui.Page) {
        if (this._useByDate.value) {
            this._setManufacturedDate(this._useByDate.value, 'useByDate');
        }
    }

    public onValidationShipByDate(value: string): string | undefined {
        if (
            value &&
            this._detentionDate.value &&
            this._shipByDate.value &&
            this._detentionDate.value >= this._shipByDate.value
        ) {
            return this._errorMessages.shipByDateError;
        }
        return undefined;
    }

    public async onChangeShipByDate(pageInstance: ui.Page) {
        if (this._shipByDate.value) {
            this._setManufacturedDate(this._shipByDate.value, 'shipByDate');
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }

    // eslint-disable-next-line class-methods-use-this
    public onValidationFifoDate(_value: string): string | undefined {
        return undefined;
    }

    public async onChangeFifoDate(pageInstance: ui.Page) {
        if (this._fifoDate.value) {
            switch (this._productConfiguration?.fifoDateNature) {
                case 'input':
                    // nothing to do in this case
                    break;

                case 'manufacturing':
                    if (!this._manufacturedDate.value) {
                        this._setStandardDate(this._fifoDate.value);
                    }
                    break;

                case 'sellByDate':
                    if (this._sellByDate.value) {
                        this._setManufacturedDate(this._sellByDate.value, 'sellByDate');
                    }
                    break;

                case 'maximumConservationLimit':
                    if (this._useByDate.value) {
                        this._setManufacturedDate(this._useByDate.value, 'useByDate');
                    }
                    break;

                case 'sendByDateLimit':
                    if (this._shipByDate.value) {
                        this._setManufacturedDate(this._shipByDate.value, 'shipByDate');
                    }
                    break;

                default:
                    return;
            }
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }

    /**
     * This function assign the FIFO date based on the current date and the product configuration.
     * @param currentDate
     * @param fifoDateNature
     * @returns
     */
    private _setFifoDate(currentDate: string, fifoDateNature?: string): boolean {
        if (!this._fifoDate.value && fifoDateNature && fifoDateNature === this._productConfiguration?.fifoDateNature) {
            this._fifoDate.value = currentDate;
            return true;
        }
        return false;
    }

    /**
     * This function calculates the manufacturing date and then readjusts the fifo date.
     * @param currentDate
     * @param fieldName
     * @returns
     */
    private _setManufacturedDate(currentDate: string, fieldName: string): boolean {
        const _currentDate = currentDate ? DateValue.parse(currentDate) : undefined;
        if (!this._manufacturedDate.value && _currentDate) {
            const _getManufacturedDate = this._dateCalculation(_currentDate, fieldName, this._productConfiguration);
            if (_getManufacturedDate) {
                this._manufacturedDate.value = _getManufacturedDate;
                this._setFifoDate(_getManufacturedDate, 'manufacturing');
                return this._setStandardDate(_getManufacturedDate);
            }
        }
        return false;
    }

    /**
     * Assign all date after change date
     * @param currentDate
     * @returns
     */
    private _setStandardDate(currentDate: string): boolean {
        const _currentDate = currentDate ? DateValue.parse(currentDate) : undefined;
        const { _productConfiguration } = this;
        if (!_productConfiguration || !_currentDate) {
            return false;
        }

        // Define date targets with their corresponding fields and FIFO nature
        const dateTargets = [
            { target: 'detentionDate', field: this._detentionDate, fifoDateNature: undefined },
            { target: 'sellByDate', field: this._sellByDate, fifoDateNature: 'sellByDate' },
            { target: 'useByDate', field: this._useByDate, fifoDateNature: 'maximumConservationLimit' },
            { target: 'shipByDate', field: this._shipByDate, fifoDateNature: 'sendByDateLimit' },
        ];

        // Process each date target using forEach instead of for loop
        dateTargets.forEach(({ target, field, fifoDateNature }) => {
            if (!field.value) {
                // extract date field parameters
                const _dateCalculated = this._dateCalculation(_currentDate, target, _productConfiguration);
                if (_dateCalculated) {
                    // set date value
                    field.value = _dateCalculated;
                    this._setFifoDate(_dateCalculated, fifoDateNature);
                }
            }
        });

        return true;
    }

    /**
     * This function calculates the date based on the current date and the product configuration.
     *  @param currentDate
     *  @param dateTarget
     *  @param productConfiguration
     *  @returns
     */
    // eslint-disable-next-line class-methods-use-this
    private _dateCalculation(
        currentDate: DateValue,
        dateTarget: string,
        productConfiguration: ProductConfiguration | undefined,
    ): string | undefined {
        if (!productConfiguration || !currentDate) {
            return undefined;
        }

        // Define configuration mapping for each date target
        const dateConfigurations = {
            detentionDate: {
                length: productConfiguration.detentionLength,
                unit: productConfiguration.detentionUnit ?? 'no',
            },
            sellByDate: {
                length: productConfiguration.sellByDateLength,
                unit: productConfiguration.sellByDateUnit ?? 'no',
            },
            useByDate: {
                length: productConfiguration.useByDateLength,
                unit: productConfiguration.useByDateUnit ?? 'no',
            },
            shipByDate: {
                length: productConfiguration.sendByDateLength,
                unit: productConfiguration.sendByDateUnit ?? 'no',
            },
        };

        const config = dateConfigurations[dateTarget as keyof typeof dateConfigurations];
        if (!config) {
            return undefined;
        }

        const { length: _dateLength, unit: _dateUnit } = config;

        // Compute date value based on unit
        switch (_dateUnit as UnitLength) {
            case 'days':
                return currentDate.addDays(_dateLength).toString();

            case 'month':
                return currentDate.addMonths(_dateLength).toString();

            default:
                return currentDate.toString();
        }
    }
}
