"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProductPalletizationPlan = void 0;
const typesLib = require("@sage/xtrem-decimal");
const storage_properties_1 = require("@sage/wh-master-data/lib/client-functions/storage-properties");
const interfaces_1 = require("@sage/wh-master-data/lib/interfaces");
const math_1 = require("@sage/wh-master-data/lib/shared-functions/math");
const ui = require("@sage/xtrem-ui");
const get_consumption_unit_1 = require("./get-consumption-unit");
const get_product_configuration_1 = require("./get-product-configuration");
class ProductPalletizationPlan {
    constructor(_inputContainer, _numberOfContainers, _homogeneousContainer, _homogeneousQuantity) {
        this._inputContainer = _inputContainer;
        this._numberOfContainers = _numberOfContainers;
        this._homogeneousContainer = _homogeneousContainer;
        this._homogeneousQuantity = _homogeneousQuantity;
        // Product palletization plan
        this._productPalletizationPlanUnits = [];
    }
    /**
     * Reinitialize all values to undefined or empty array
     */
    reinitialize() {
        this._siteCode = undefined;
        this._depositorCode = undefined;
        this._productConfiguration = undefined;
        this._productPalletizationPlanUnits = [];
        this._productConsumptionUnit = undefined;
    }
    /**
     * Initialize container units : undefined values for reset values
     * @param pageInstance
     * @param productCode
     * @param maxRecords
     * @returns
     */
    async initialize(pageInstance, productCode, maxRecords = 100) {
        const _selectedSiteDepositor = (0, storage_properties_1.getSelectedSiteDepositor)(pageInstance);
        const _siteCode = _selectedSiteDepositor?.site;
        const _depositorCode = _selectedSiteDepositor?.depositor;
        if (_siteCode && _depositorCode && productCode) {
            this._siteCode = _siteCode;
            this._depositorCode = _depositorCode;
            this._productConfiguration = await (0, get_product_configuration_1.getProductConfiguration)(pageInstance, productCode);
            if (!this._productConfiguration) {
                this.reinitialize();
                return false;
            }
            const _productContainerUnitList = await (0, get_consumption_unit_1.getProductContainersUnits)(pageInstance, productCode, typesLib.lt(maxRecords, 100) ? 100 : maxRecords);
            this._productPalletizationPlanUnits = _productContainerUnitList;
            this._productConsumptionUnit = _productContainerUnitList.find(_ => typesLib.strictEq(_.containerLevel, interfaces_1.hightestContainerLevel));
            return typesLib.gt(this._productPalletizationPlanUnits.length, 0);
        }
        this.reinitialize();
        return false;
    }
    /**
     * Get site code
     */
    get siteCode() {
        return this._siteCode ?? '';
    }
    /**
     * Get depositor code
     */
    get depositorCode() {
        return this._depositorCode ?? '';
    }
    /**
     * Get product code
     */
    get productCode() {
        return this._productConfiguration?.code ?? '';
    }
    /**
     *  Get product configuration
     */
    get productConfiguration() {
        return this._productConfiguration ? { ...this._productConfiguration } : undefined;
    }
    /**
     * Get product consumption unit code
     * @returns consumption unit
     */
    get productConsumptionUnit() {
        return this._productConsumptionUnit;
    }
    /**
     * Get product consumption unit code
     */
    get productConsumptionUnitCode() {
        return this._productConsumptionUnit?.code ?? '';
    }
    /**
     *  Get product consumption unit stock unit
     */
    get productConsumptionUnitStockUnit() {
        return this._productConsumptionUnit?.stockUnit;
    }
    /**
     * Get product consumption unit precision
     */
    get productConsumptionUnitPrecision() {
        return this._productConsumptionUnit?.numberOfDecimals;
    }
    /**
     *  Get product default container unit
     */
    get productDefaultContainer() {
        return this.getProductContainerUnit(this._productConfiguration?.defaultContainerCode);
    }
    /**
     *  Get product default container code
     */
    get productDefaultContainerCode() {
        return this.productDefaultContainer?.code ?? '';
    }
    /**
     * Get product palletization plan units
     */
    get productPalletizationPlanUnits() {
        return this._productPalletizationPlanUnits;
    }
    /**
     * Calculate the possible movement plan :
     *  - if no inputContainerCode is provided, we search for the most appropriate container and we calculate the complete plan.
     *  - if the number of containers is empty, we determine it.
     *  - if the homogeneous container is not provided, we take the lower container of the input container or itself if level 5.
     *
     * The calculation will be done from these elements.
     * @param numberOfConsumptionUnit
     * @param inputContainerCode
     * @param numberOfContainers
     * @param homogeneousContainerCode
     * @param allowChangeHomogeneousContainer optional to allow change homogeneous container when no valid plan is found
     * @returns movement plan or undefined
     */
    calculatePossibleMovementPlan(numberOfConsumptionUnit, inputContainerCode, numberOfContainers, homogeneousContainerCode, allowChangeHomogeneousContainer = true) {
        let _inputContainerCode;
        let _numberOfContainers = numberOfContainers;
        let _homogeneousContainerCode = _inputContainerCode ? homogeneousContainerCode : undefined;
        const _containerUnit = this.getProductContainerUnit(inputContainerCode) ??
            this.getHighestProductContainerUnit(numberOfConsumptionUnit);
        if (_containerUnit) {
            _inputContainerCode = _containerUnit.code;
            _homogeneousContainerCode =
                typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel) ? _inputContainerCode
                    : homogeneousContainerCode;
            if (!_homogeneousContainerCode) {
                const _homogeneousContainerUnit = this.getNearProductContainerUnit(numberOfConsumptionUnit, _inputContainerCode);
                _homogeneousContainerCode = _homogeneousContainerUnit?.code ?? _inputContainerCode;
            }
            _numberOfContainers =
                inputContainerCode && numberOfContainers
                    ? numberOfContainers
                    : this.getNumberOfContainers(numberOfConsumptionUnit, _inputContainerCode);
            const _numberOfHomogeneousContainers = typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel) ? 1
                : this.getNumberOfHomogeneousContainers(numberOfConsumptionUnit, _inputContainerCode, _numberOfContainers, _homogeneousContainerCode ?? '');
            /**
             * If the provided homogeneous container cannot be used, but the search for an alternative
             * has been authorized, an attempt is made to find the closest possible one.
             */
            if (!_numberOfHomogeneousContainers && allowChangeHomogeneousContainer && homogeneousContainerCode) {
                return this.calculatePossibleMovementPlan(numberOfConsumptionUnit, _inputContainerCode, _numberOfContainers, undefined, false);
            }
            return {
                inputContainerCode: _inputContainerCode,
                homogeneousContainerCode: _homogeneousContainerCode,
                numberOfContainers: _numberOfContainers,
                numberOfHomogeneousContainers: _numberOfHomogeneousContainers,
            };
        }
        return undefined;
    }
    /**
     * Convert container to consumption quantity
     * @param containerCode
     * @param numberOfContainer
     * @param homogeneousContainerCode
     * @param homogeneousQuantity
     * @returns consumption quantity
     * */
    convertContainerPlanToConsumptionQuantity(containerCode, numberOfContainer, homogeneousContainerCode, homogeneousQuantity) {
        if (containerCode && homogeneousContainerCode) {
            const _productConsumptionUnit = this.productConsumptionUnit;
            if (_productConsumptionUnit) {
                /**
                 * When the container is of the consumption unit (level 5), we do not need
                 * to take into account the homogeneous container because it is identical.
                 */
                if (typesLib.strictEq(_productConsumptionUnit.code, containerCode)) {
                    return (0, math_1.truncate)(numberOfContainer, _productConsumptionUnit.numberOfDecimals);
                }
                const _homogeneousContainerUnit = this.getProductContainerUnit(homogeneousContainerCode);
                if (_homogeneousContainerUnit) {
                    return (0, math_1.truncate)(typesLib.mul(typesLib.mul(numberOfContainer, homogeneousQuantity), _homogeneousContainerUnit.numberOfConsumptionUnit), _productConsumptionUnit.numberOfDecimals);
                }
            }
        }
        return 0;
    }
    /**
     *  Convert quantity in consumption unit
     * @param containerQuantity
     * @param containerCode
     * @returns
     */
    convertContainerQuantityInConsumptionUnit(containerQuantity, containerCode) {
        return this.truncateConsumptionQuantity(typesLib.mul(containerQuantity, this.getProductContainerNumberOfConsumptionUnit(containerCode)));
    }
    /**
     *  Convert quantity from consumption unit
     * @param quantityInConsumptionUnit
     * @param containerCode
     * @returns
     */
    convertQuantityInConsumptionUnitToContainer(quantityInConsumptionUnit, containerCode) {
        const _containerUnit = this.getProductContainerNumberOfConsumptionUnit(containerCode);
        return _containerUnit ? this.truncateConsumptionQuantity(typesLib.div(quantityInConsumptionUnit, _containerUnit)) : 0;
    }
    /**
     * Calculate the possible container units codes based on the input container.
     * @returns array of sorted container units codes
     */
    getHomogeneousProductContainerUnitOptions() {
        const _containerUnitOptions = [];
        let _containerUnit = this.getProductContainerUnit(this._inputContainer.value);
        if (!_containerUnit) {
            return [];
        }
        // Limit iterations to prevent infinite loops
        const maxIterations = 10;
        let iterations = 0;
        while (_containerUnit && typesLib.lt(iterations, maxIterations)) {
            _containerUnitOptions.push(_containerUnit.code);
            if (typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
                break;
            }
            const _inferiorContainerCode = _containerUnit.inferiorContainer?.container?.code;
            if (!_inferiorContainerCode) {
                break;
            }
            _containerUnit = this._productPalletizationPlanUnits.find(unit => typesLib.strictEq(unit.code, _inferiorContainerCode));
            iterations = typesLib.add(iterations, 1);
        }
        // Sort by level first (level1, level2, ..., level5), then by container code
        return _containerUnitOptions.sort((a, b) => {
            const unitA = this._productPalletizationPlanUnits.find(unit => typesLib.strictEq(unit.code, a));
            const unitB = this._productPalletizationPlanUnits.find(unit => typesLib.strictEq(unit.code, b));
            if (unitA && unitB) {
                // Sort by level first, then by code (level1 < level2 < ... < level5 naturally)
                if (typesLib.strictNe(unitA.containerLevel, unitB.containerLevel)) {
                    return typesLib.lt(unitA.containerLevel, unitB.containerLevel) ? -1 : 1;
                }
                if (typesLib.lt(a, b))
                    return -1;
                if (typesLib.gt(a, b))
                    return 1;
                return 0;
            }
            // Fallback to alphabetical sort if units not found
            if (typesLib.lt(a, b))
                return -1;
            if (typesLib.gt(a, b))
                return 1;
            return 0;
        });
    }
    /**
     * Get product container unit options
     * @returns array of sorted container units codes (sorted by level then code)
     */
    getProductContainerUnitOptions() {
        return this._productPalletizationPlanUnits
            .sort((a, b) => {
            // Sort by level first, then by container code (level1 < level2 < ... < level5 naturally)
            if (typesLib.strictNe(a.containerLevel, b.containerLevel)) {
                return typesLib.lt(a.containerLevel, b.containerLevel) ? -1 : 1;
            }
            if (typesLib.lt(a.code, b.code))
                return -1;
            if (typesLib.gt(a.code, b.code))
                return 1;
            return 0;
        })
            .map(productContainerUnit => productContainerUnit.code);
    }
    /**
     * Get container unit
     * @param containerCode
     * @returns container unit or undefined
     */
    getProductContainerUnit(containerCode) {
        return this._productPalletizationPlanUnits.find(unit => typesLib.strictEq(unit.code, containerCode));
    }
    /**
     * Get product container (without any reference and number of consumption unit as string)
     * @param containerCode
     * @returns product container unit or undefined
     * */
    getProductContainer(containerCode) {
        const _containerUnit = this.getProductContainerUnit(containerCode);
        return {
            ..._containerUnit,
            numberOfConsumptionUnit: String(_containerUnit?.numberOfConsumptionUnit),
        };
    }
    /**
     * Get number of consumption unit
     * @param containerCode
     * @returns number of consumption unit or 0
     */
    getProductContainerNumberOfConsumptionUnit(containerCode) {
        return Number(this.getProductContainerUnit(containerCode)?.numberOfConsumptionUnit);
    }
    /**
     * Get container unit depending on barcode and
     * @param defaultMode
     * @param inputContainerCode
     * @returns container unit and options
     */
    getProductContainerUnitDependingBarCode(defaultMode, inputContainerCode) {
        let _containerUnit;
        let _containerOptions = [];
        if (typesLib.strictEq(defaultMode, 'eanCode')) {
            _containerUnit = this._productPalletizationPlanUnits.find(_ => typesLib.strictEq(_.code, inputContainerCode));
            _containerOptions = _containerUnit?.code ? [_containerUnit?.code] : [];
        }
        else {
            _containerUnit = this.productConsumptionUnit;
            _containerOptions = this.getProductContainerUnitOptions();
        }
        return _containerUnit
            ? { containerUnit: _containerUnit, containerOptions: _containerOptions }
            : undefined;
    }
    getContainerUnit(quantityInConsumptionUnit) {
        return this._productPalletizationPlanUnits.find(unit => typesLib.gte(quantityInConsumptionUnit, unit.numberOfConsumptionUnit));
    }
    /**
     *  Indicates whether the product is managed in stock unit
     * @returns boolean
     */
    getIsProductByStockUnit() {
        return !!this.productConsumptionUnitStockUnit;
    }
    /**
     * Get product container unit precision
     * Only an item managed in stock unit and whose container is level 5 (consumption unit)
     * sees the quantity of the latter with a precision which can be different from zero.
     * @param containerCode default is product consumption unit code
     * @returns precision or 0
     */
    getProductContainerUnitPrecision(containerCode) {
        const _containerCode = containerCode ?? this.productConsumptionUnitCode;
        return this.getIsProductByStockUnit() && typesLib.strictEq(_containerCode, this.productConsumptionUnitCode)
            ? (this.productConsumptionUnitPrecision ?? 0)
            : 0;
    }
    /**
     * Search the palletization plan for the most suitable container for the proposed quantity.
     * @param quantityInConsumptionUnit
     * @returns container unit or undefined
     */
    getHighestProductContainerUnit(quantityInConsumptionUnit) {
        try {
            // Filter containers that can accommodate the quantity and sort by level then capacity
            const validContainers = this._productPalletizationPlanUnits
                .filter(unit => typesLib.gte(unit.numberOfConsumptionUnit, quantityInConsumptionUnit))
                .sort((a, b) => {
                // Sort by level first (level1 < level2 < ... < level5 naturally), then by quantity desc
                if (typesLib.strictNe(a.containerLevel, b.containerLevel)) {
                    return typesLib.lt(a.containerLevel, b.containerLevel) ? -1 : 1;
                }
                return typesLib.sub(b.numberOfConsumptionUnit, a.numberOfConsumptionUnit);
            });
            // Return the first valid container (lowest level, highest capacity within that level)
            if (typesLib.gt(validContainers.length, 0)) {
                return validContainers[0];
            }
            /**
             * If no container is found, we return the container with the highest capacity
             */
            return this._productPalletizationPlanUnits.reduce((prev, current) => {
                return !prev || typesLib.gt(current.numberOfConsumptionUnit, prev.numberOfConsumptionUnit) ? current : prev;
            }, undefined);
        }
        catch (error) {
            ui.console.error(`Error in getHighestProductContainerUnit:\n${error}`);
        }
        return undefined;
    }
    /**
     * Search for the homogeneous container associated with this level; the latter cannot be higher.
     * @param inputContainerCode
     * @returns
     */
    getHomogeneousProductContainerUnit(inputContainerCode) {
        let _containerUnit = this.getProductContainerUnit(inputContainerCode);
        const _homogeneousLevel = _containerUnit?.homogeneousLevel;
        // Limit iterations to prevent infinite loops
        const maxIterations = 10;
        let iterations = 0;
        while (_containerUnit && typesLib.lt(iterations, maxIterations)) {
            if (typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel) || typesLib.strictEq(_containerUnit.containerLevel, _homogeneousLevel)) {
                return _containerUnit;
            }
            const inferiorContainerCode = _containerUnit.inferiorContainer?.container?.code;
            if (!inferiorContainerCode) {
                break;
            }
            _containerUnit = this._productPalletizationPlanUnits.find(unit => typesLib.strictEq(unit.code, inferiorContainerCode));
            iterations = typesLib.add(iterations, 1);
        }
        return undefined;
    }
    /**
     * Get the code of the homogeneous product container associated with the input container code.
     * @param inputContainerCode
     * @returns code of homogeneous container
     */
    getHomogeneousProductContainerUnitCode(inputContainerCode) {
        return this.getHomogeneousProductContainerUnit(inputContainerCode)?.code ?? '';
    }
    /**
     *  Get number of consumption unit of homogeneous container
     * @param inputContainerCode
     * @returns number of consumption unit
     */
    getHomogeneousProductContainerUnitQuantity(inputContainerCode) {
        const _homogeneousContainerUnit = this.getHomogeneousProductContainerUnit(inputContainerCode);
        return _homogeneousContainerUnit?.numberOfConsumptionUnit ?? 0;
    }
    /**
     *  Get the multiple of homogeneous container for a given quantity in consumption unit if quantity is not multiple
     * @param quantityInConsumptionUnit
     * @param inputContainerCode
     * @param homogeneousContainerCode
     * @returns 0 when quantity is multiple else return multiple
     */
    getHomogeneousProductContainerUnitMultiple(quantityInConsumptionUnit, inputContainerCode, homogeneousContainerCode, isAllowSameContainer) {
        // The input and homogeneous containers must be valid.
        if (!this.validateContainerUnit(inputContainerCode, homogeneousContainerCode, isAllowSameContainer)) {
            return 0;
        }
        // We determine the homogeneous level container
        const _homogeneousLevelContainerUnit = this.getHomogeneousProductContainerUnit(inputContainerCode);
        if (!_homogeneousLevelContainerUnit || typesLib.strictEq(_homogeneousLevelContainerUnit.code, homogeneousContainerCode) || typesLib.strictEq(_homogeneousLevelContainerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
            return 0;
        }
        const _homogeneousLevelQuantity = _homogeneousLevelContainerUnit.numberOfConsumptionUnit;
        if (typesLib.strictEq(_homogeneousLevelQuantity, 1)) {
            return 0;
        }
        const _remainder = quantityInConsumptionUnit % _homogeneousLevelQuantity;
        return _remainder ? _homogeneousLevelQuantity : 0;
    }
    /**
     *  Get the container unit near the quantity in consumption unit
     *  It must be specified whether the parent should also be considered in the search.
     * @param quantityInConsumptionUnit
     * @param inputContainerCode
     * @param includesInputContainer
     * @returns
     */
    getNearProductContainerUnit(quantityInConsumptionUnit, inputContainerCode, includesInputContainer) {
        let _containerUnit = this.getProductContainerUnit(inputContainerCode);
        // Limit iterations to prevent infinite loops
        const maxIterations = 10;
        let iterations = 0;
        while (_containerUnit && typesLib.lt(iterations, maxIterations)) {
            if (typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel) ||
                (typesLib.lte(_containerUnit.numberOfConsumptionUnit, quantityInConsumptionUnit) &&
                    (typesLib.strictNe(inputContainerCode, _containerUnit.code) || includesInputContainer))) {
                return _containerUnit;
            }
            const inferiorContainerCode = _containerUnit.inferiorContainer?.container?.code;
            if (!inferiorContainerCode) {
                break;
            }
            _containerUnit = this._productPalletizationPlanUnits.find(unit => typesLib.strictEq(unit.code, inferiorContainerCode));
            iterations = typesLib.add(iterations, 1);
        }
        return _containerUnit ?? this.productConsumptionUnit;
    }
    /**
     * Get the number of containers for a given quantity in consumption unit
     * @param containerUnitCode
     * @param quantityInConsumptionUnit
     * @returns number of containers
     */
    getNumberOfContainers(quantityInConsumptionUnit, containerUnitCode) {
        if (typesLib.gt(quantityInConsumptionUnit, 0)) {
            const _containerUnit = containerUnitCode
                ? this.getProductContainerUnit(containerUnitCode)
                : this.getHighestProductContainerUnit(quantityInConsumptionUnit);
            if (_containerUnit) {
                if (typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
                    if (_containerUnit?.stockUnit) {
                        return typesLib.max(this.truncateConsumptionQuantity(quantityInConsumptionUnit), 1);
                    }
                    return typesLib.max(Math.trunc(quantityInConsumptionUnit), 1);
                }
                return typesLib.max(Math.trunc(typesLib.div(quantityInConsumptionUnit, _containerUnit.numberOfConsumptionUnit)), 1);
            }
        }
        return 0;
    }
    /**
     * Get the number of homogeneous containers for a given quantity in consumption unit
     * @param containerUnitCode
     * @param numberOfContainers
     * @param quantityInConsumptionUnit
     * @param homogeneousContainerCode
     * @returns number of homogeneous containers
     */
    getNumberOfHomogeneousContainers(quantityInConsumptionUnit, containerUnitCode, numberOfContainers, homogeneousContainerCode) {
        if (typesLib.gt(numberOfContainers, 0) && typesLib.gt(quantityInConsumptionUnit, 0)) {
            const _containerUnit = this.getProductContainerUnit(containerUnitCode);
            const _homogeneousContainerUnit = typesLib.strictEq(containerUnitCode, homogeneousContainerCode) ? _containerUnit
                : this.getProductContainerUnit(homogeneousContainerCode);
            if (_homogeneousContainerUnit && _containerUnit) {
                if (typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
                    if (typesLib.strictNe(containerUnitCode, homogeneousContainerCode)) {
                        return 0;
                    }
                    if (_containerUnit?.stockUnit) {
                        return this.truncateConsumptionQuantity(typesLib.div(quantityInConsumptionUnit, numberOfContainers));
                    }
                    return Math.trunc(typesLib.div(quantityInConsumptionUnit, numberOfContainers));
                }
                if (typesLib.strictEq(_homogeneousContainerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
                    if (_homogeneousContainerUnit?.stockUnit) {
                        return this.truncateConsumptionQuantity(typesLib.div(quantityInConsumptionUnit, numberOfContainers));
                    }
                    return typesLib.min(Math.trunc(typesLib.div(quantityInConsumptionUnit, numberOfContainers)), _containerUnit.numberOfConsumptionUnit);
                }
                return Math.trunc(typesLib.div(quantityInConsumptionUnit, (typesLib.mul(_homogeneousContainerUnit.numberOfConsumptionUnit, numberOfContainers))));
            }
        }
        return 0;
    }
    /**
     * truncate consumption quantity
     * @param quantity
     * @returns
     */
    truncateConsumptionQuantity(quantity) {
        return (0, math_1.truncate)(quantity, this.productConsumptionUnitPrecision ?? 0);
    }
    /**
     * Validate container unit
     * @param inputContainerCode
     * @param homogeneousContainerCode
     * @param isAllowSameContainer
     * @returns boolean
     */
    validateContainerUnit(inputContainerCode, homogeneousContainerCode, isAllowSameContainer) {
        if (homogeneousContainerCode && inputContainerCode) {
            const _inputContainerUnit = this.getProductContainerUnit(inputContainerCode);
            if (_inputContainerUnit) {
                if (isAllowSameContainer && typesLib.strictEq(homogeneousContainerCode, inputContainerCode)) {
                    return true;
                }
                // Check if the input container is the highest container level
                if (typesLib.strictEq(_inputContainerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
                    if (typesLib.strictNe(homogeneousContainerCode, inputContainerCode)) {
                        return false;
                    }
                }
                else {
                    //  Check if the homogeneous container is higher than the input container
                    const _homogeneousContainerUnit = this.getProductContainerUnit(homogeneousContainerCode);
                    if (_homogeneousContainerUnit) {
                        if (typesLib.gte(_inputContainerUnit?.containerLevel, _homogeneousContainerUnit?.containerLevel)) {
                            return false;
                        }
                        /**
                         * Check that the homogeneous container is indeed the child of the
                         * containing input container in the palletization plan
                         */
                        if (typesLib.strictNe(_homogeneousContainerUnit?.containerLevel, interfaces_1.hightestContainerLevel)) {
                            let _inferiorContainerUnit = _inputContainerUnit;
                            while (_inferiorContainerUnit) {
                                const _inferiorContainerUnitCode = _inferiorContainerUnit.inferiorContainer?.container?.code;
                                // Check if the homogeneous container is the child of the inferior container
                                if (typesLib.strictEq(_inferiorContainerUnitCode, homogeneousContainerCode)) {
                                    break;
                                }
                                if (!_inferiorContainerUnitCode) {
                                    return false;
                                }
                                _inferiorContainerUnit = this.getProductContainerUnit(_inferiorContainerUnitCode);
                                if (!_inferiorContainerUnit || typesLib.lt(_homogeneousContainerUnit?.containerLevel, _inferiorContainerUnit?.containerLevel)) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }
        }
        return true;
    }
    /**
     * On change input container event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    async onChangeInputContainer(pageInstance, numberOfConsumptionUnit) {
        // inputContainer, numberOfContainers, homogeneousContainer, homogeneousQuantity, packedQuantity
        if (this._inputContainer.value) {
            const _containerUnit = this.getProductContainerUnit(this._inputContainer.value);
            if (_containerUnit?.code) {
                this._numberOfContainers.value = this.getNumberOfContainers(numberOfConsumptionUnit, _containerUnit.code);
                if (typesLib.strictEq(_containerUnit.containerLevel, interfaces_1.hightestContainerLevel)) {
                    this._homogeneousContainer.value = this._inputContainer.value;
                    this._homogeneousContainer.isDisabled = true;
                    this._homogeneousQuantity.value = 1;
                    this._homogeneousQuantity.isDisabled = true;
                }
                else {
                    // near the lowest container level
                    const _inferiorContainerUnitCode = this.getNearProductContainerUnit(numberOfConsumptionUnit, _containerUnit.code)?.code ?? null;
                    this._homogeneousContainer.isDisabled = false;
                    this._homogeneousQuantity.isDisabled = false;
                    this._homogeneousContainer.value = _inferiorContainerUnitCode;
                    this._homogeneousQuantity.value = this.getNumberOfHomogeneousContainers(numberOfConsumptionUnit, _containerUnit.code, Number(this._numberOfContainers.value), this._homogeneousContainer.value);
                }
                await pageInstance.$.commitValueAndPropertyChanges();
            }
        }
    }
    /**
     * On change number of containers event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    async onChangeNumberOfContainers(pageInstance, numberOfConsumptionUnit) {
        await this.onChangeHomogeneousContainer(pageInstance, numberOfConsumptionUnit);
    }
    /**
     * On change homogeneous container event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    async onChangeHomogeneousContainer(pageInstance, numberOfConsumptionUnit) {
        if (this._inputContainer.value &&
            this._homogeneousContainer.value &&
            !this._homogeneousContainer.isDisabled &&
            !this._homogeneousQuantity.isDisabled) {
            this._homogeneousQuantity.value = this.getNumberOfHomogeneousContainers(numberOfConsumptionUnit, this._inputContainer.value, Number(this._numberOfContainers.value), this._homogeneousContainer.value);
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
}
exports.ProductPalletizationPlan = ProductPalletizationPlan;
//# sourceMappingURL=product-palletization-plan.js.map