import type { GraphApi } from '@sage/wh-product-data-api';
import * as ui from '@sage/xtrem-ui';
import { ContainerProductUnit, ContainerUnit, ContainerUnitAndOptions, ContainerUnits, MovementPlan, ProductConfiguration } from '../interfaces/environnement';
export declare class ProductPalletizationPlan {
    private _inputContainer;
    private _numberOfContainers;
    private _homogeneousContainer;
    private _homogeneousQuantity;
    constructor(_inputContainer: ui.fields.DropdownList, _numberOfContainers: ui.fields.Numeric, _homogeneousContainer: ui.fields.DropdownList, _homogeneousQuantity: ui.fields.Numeric);
    private _siteCode?;
    private _depositorCode?;
    private _productConfiguration?;
    private _productPalletizationPlanUnits;
    private _productConsumptionUnit?;
    /**
     * Reinitialize all values to undefined or empty array
     */
    reinitialize(): void;
    /**
     * Initialize container units : undefined values for reset values
     * @param pageInstance
     * @param productCode
     * @param maxRecords
     * @returns
     */
    initialize(pageInstance: ui.Page<GraphApi>, productCode: string | undefined | null, maxRecords?: number): Promise<boolean>;
    /**
     * Get site code
     */
    get siteCode(): string;
    /**
     * Get depositor code
     */
    get depositorCode(): string;
    /**
     * Get product code
     */
    get productCode(): string;
    /**
     *  Get product configuration
     */
    get productConfiguration(): ProductConfiguration | undefined;
    /**
     * Get product consumption unit code
     * @returns consumption unit
     */
    get productConsumptionUnit(): ContainerUnit | undefined;
    /**
     * Get product consumption unit code
     */
    get productConsumptionUnitCode(): string;
    /**
     *  Get product consumption unit stock unit
     */
    get productConsumptionUnitStockUnit(): string | undefined;
    /**
     * Get product consumption unit precision
     */
    get productConsumptionUnitPrecision(): number | undefined;
    /**
     *  Get product default container unit
     */
    get productDefaultContainer(): ContainerUnit | undefined;
    /**
     *  Get product default container code
     */
    get productDefaultContainerCode(): string;
    /**
     * Get product palletization plan units
     */
    get productPalletizationPlanUnits(): ContainerUnits;
    /**
     * Calculate the possible movement plan :
     *  - if no inputContainerCode is provided, we search for the most appropriate container and we calculate the complete plan.
     *  - if the number of containers is empty, we determine it.
     *  - if the homogeneous container is not provided, we take the lower container of the input container or itself if level 5.
     *
     * The calculation will be done from these elements.
     * @param numberOfConsumptionUnit
     * @param inputContainerCode
     * @param numberOfContainers
     * @param homogeneousContainerCode
     * @param allowChangeHomogeneousContainer optional to allow change homogeneous container when no valid plan is found
     * @returns movement plan or undefined
     */
    calculatePossibleMovementPlan(numberOfConsumptionUnit: number, inputContainerCode: string | undefined | null, numberOfContainers: number, homogeneousContainerCode: string | undefined | null, allowChangeHomogeneousContainer?: boolean): MovementPlan | undefined;
    /**
     * Convert container to consumption quantity
     * @param containerCode
     * @param numberOfContainer
     * @param homogeneousContainerCode
     * @param homogeneousQuantity
     * @returns consumption quantity
     * */
    convertContainerPlanToConsumptionQuantity(containerCode: string | undefined | null, numberOfContainer: number, homogeneousContainerCode: string | undefined | null, homogeneousQuantity: number): number;
    /**
     *  Convert quantity in consumption unit
     * @param containerQuantity
     * @param containerCode
     * @returns
     */
    convertContainerQuantityInConsumptionUnit(containerQuantity: number, containerCode: string | undefined | null): number;
    /**
     *  Convert quantity from consumption unit
     * @param quantityInConsumptionUnit
     * @param containerCode
     * @returns
     */
    convertQuantityInConsumptionUnitToContainer(quantityInConsumptionUnit: number, containerCode: string | undefined | null): number;
    /**
     * Calculate the possible container units codes based on the input container.
     * @returns array of sorted container units codes
     */
    getHomogeneousProductContainerUnitOptions(): string[];
    /**
     * Get product container unit options
     * @returns array of sorted container units codes (sorted by level then code)
     */
    getProductContainerUnitOptions(): string[];
    /**
     * Get container unit
     * @param containerCode
     * @returns container unit or undefined
     */
    getProductContainerUnit(containerCode: string | undefined | null): ContainerUnit | undefined;
    /**
     * Get product container (without any reference and number of consumption unit as string)
     * @param containerCode
     * @returns product container unit or undefined
     * */
    getProductContainer(containerCode: string | undefined | null): (ContainerProductUnit & {
        numberOfConsumptionUnit: string;
    }) | undefined;
    /**
     * Get number of consumption unit
     * @param containerCode
     * @returns number of consumption unit or 0
     */
    getProductContainerNumberOfConsumptionUnit(containerCode: string | undefined | null): number;
    /**
     * Get container unit depending on barcode and
     * @param defaultMode
     * @param inputContainerCode
     * @returns container unit and options
     */
    getProductContainerUnitDependingBarCode(defaultMode: string, inputContainerCode: string | undefined | null): ContainerUnitAndOptions | undefined;
    getContainerUnit(quantityInConsumptionUnit: number): ContainerUnit | undefined;
    /**
     *  Indicates whether the product is managed in stock unit
     * @returns boolean
     */
    getIsProductByStockUnit(): boolean;
    /**
     * Get product container unit precision
     * Only an item managed in stock unit and whose container is level 5 (consumption unit)
     * sees the quantity of the latter with a precision which can be different from zero.
     * @param containerCode default is product consumption unit code
     * @returns precision or 0
     */
    getProductContainerUnitPrecision(containerCode?: string | null): number;
    /**
     * Search the palletization plan for the most suitable container for the proposed quantity.
     * @param quantityInConsumptionUnit
     * @returns container unit or undefined
     */
    getHighestProductContainerUnit(quantityInConsumptionUnit: number): ContainerUnit | undefined;
    /**
     * Search for the homogeneous container associated with this level; the latter cannot be higher.
     * @param inputContainerCode
     * @returns
     */
    getHomogeneousProductContainerUnit(inputContainerCode: string): ContainerUnit | undefined;
    /**
     * Get the code of the homogeneous product container associated with the input container code.
     * @param inputContainerCode
     * @returns code of homogeneous container
     */
    getHomogeneousProductContainerUnitCode(inputContainerCode: string): string;
    /**
     *  Get number of consumption unit of homogeneous container
     * @param inputContainerCode
     * @returns number of consumption unit
     */
    getHomogeneousProductContainerUnitQuantity(inputContainerCode: string): number;
    /**
     *  Get the multiple of homogeneous container for a given quantity in consumption unit if quantity is not multiple
     * @param quantityInConsumptionUnit
     * @param inputContainerCode
     * @param homogeneousContainerCode
     * @returns 0 when quantity is multiple else return multiple
     */
    getHomogeneousProductContainerUnitMultiple(quantityInConsumptionUnit: number, inputContainerCode: string, homogeneousContainerCode: string, isAllowSameContainer?: boolean): number;
    /**
     *  Get the container unit near the quantity in consumption unit
     *  It must be specified whether the parent should also be considered in the search.
     * @param quantityInConsumptionUnit
     * @param inputContainerCode
     * @param includesInputContainer
     * @returns
     */
    getNearProductContainerUnit(quantityInConsumptionUnit: number, inputContainerCode: string, includesInputContainer?: boolean): ContainerUnit | undefined;
    /**
     * Get the number of containers for a given quantity in consumption unit
     * @param containerUnitCode
     * @param quantityInConsumptionUnit
     * @returns number of containers
     */
    getNumberOfContainers(quantityInConsumptionUnit: number, containerUnitCode: string | undefined | null): number;
    /**
     * Get the number of homogeneous containers for a given quantity in consumption unit
     * @param containerUnitCode
     * @param numberOfContainers
     * @param quantityInConsumptionUnit
     * @param homogeneousContainerCode
     * @returns number of homogeneous containers
     */
    getNumberOfHomogeneousContainers(quantityInConsumptionUnit: number, containerUnitCode: string, numberOfContainers: number, homogeneousContainerCode: string | null | undefined): number;
    /**
     * truncate consumption quantity
     * @param quantity
     * @returns
     */
    truncateConsumptionQuantity(quantity: number): number;
    /**
     * Validate container unit
     * @param inputContainerCode
     * @param homogeneousContainerCode
     * @param isAllowSameContainer
     * @returns boolean
     */
    validateContainerUnit(inputContainerCode: string | undefined | null, homogeneousContainerCode: string | undefined | null, isAllowSameContainer?: boolean): boolean;
    /**
     * On change input container event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    onChangeInputContainer(pageInstance: ui.Page<GraphApi>, numberOfConsumptionUnit: number): Promise<void>;
    /**
     * On change number of containers event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    onChangeNumberOfContainers(pageInstance: ui.Page<GraphApi>, numberOfConsumptionUnit: number): Promise<void>;
    /**
     * On change homogeneous container event
     * @param pageInstance
     * @param numberOfConsumptionUnit
     */
    onChangeHomogeneousContainer(pageInstance: ui.Page<GraphApi>, numberOfConsumptionUnit: number): Promise<void>;
}
//# sourceMappingURL=product-palletization-plan.d.ts.map