"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getConsumptionUnit = getConsumptionUnit;
exports.getConsumptionUnitCode = getConsumptionUnitCode;
exports.getProductContainersUnits = getProductContainersUnits;
const typesLib = require("@sage/xtrem-decimal");
const storage_properties_1 = require("@sage/wh-master-data/lib/client-functions/storage-properties");
const xtrem_client_1 = require("@sage/xtrem-client");
const ui = require("@sage/xtrem-ui");
/**
 * Read the product container units list
 * @param pageInstance
 * @param productCode
 * @param numberOfRecords
 * @param isConsumptionUnitOnly for read only consumption unit level
 * @returns
 */
async function _getProductContainersUnits(pageInstance, productCode, numberOfRecords = 10, isConsumptionUnitOnly = false) {
    const _selectedSiteDepositor = (0, storage_properties_1.getSelectedSiteDepositor)(pageInstance);
    const _siteCode = _selectedSiteDepositor?.site;
    const _depositorCode = _selectedSiteDepositor?.depositor;
    if (_siteCode && productCode && _depositorCode) {
        try {
            return ((0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
                .node('@sage/wh-product-data/ProductContainer')
                .query(ui.queryUtils.edgesSelector({
                product: {
                    _id: true,
                    code: true,
                    isStockUnitManagement: true,
                    stockUnit: {
                        _id: true,
                        code: true,
                        numberOfDecimals: true,
                    },
                },
                container: {
                    _id: true,
                    code: true,
                    containerManagementType: true,
                },
                containerLevel: true,
                homogeneousLevel: true,
                numberOfConsumptionUnit: true,
                inferiorContainer: {
                    _id: true,
                    containerLevel: true,
                    container: { _id: true, code: true, containerManagementType: true },
                },
                isKeyInSupportNumbers: true,
                _id: true,
            }, {
                filter: {
                    site: {
                        code: _siteCode,
                    },
                    depositor: {
                        code: _depositorCode,
                    },
                    product: { code: productCode, isActive: true },
                    isActive: true,
                    ...(isConsumptionUnitOnly && { containerLevel: 'level5' }),
                },
                first: typesLib.gt(numberOfRecords, 10) ? numberOfRecords : 10,
                orderBy: { containerLevel: -1, container: { code: 1 } },
            }))
                .execute())?.map(_ => ({
                _id: _._id,
                code: _.container?.code,
                containerLevel: _.containerLevel,
                container: {
                    _id: _.container?._id,
                    code: _.container?.code,
                    containerManagementType: _.container?.containerManagementType,
                },
                homogeneousLevel: _.homogeneousLevel,
                numberOfConsumptionUnit: Number(_.numberOfConsumptionUnit),
                isKeyInSupportNumbers: _.isKeyInSupportNumbers,
                ...(_.inferiorContainer?.container.code && {
                    inferiorContainer: {
                        _id: _.inferiorContainer._id,
                        containerLevel: _.inferiorContainer?.containerLevel,
                        container: {
                            _id: _.inferiorContainer.container._id,
                            code: _.inferiorContainer?.container.code,
                            containerManagementType: _.inferiorContainer?.container?.containerManagementType,
                        },
                    },
                }),
                ...(typesLib.strictEq(_.containerLevel, 'level5') &&
                    _.product.isStockUnitManagement &&
                    _.product?.stockUnit?.code && typesLib.strictEq(_.container?.containerManagementType, 'unit') && {
                    _id: _.product.stockUnit._id,
                    stockUnit: _.product.stockUnit.code,
                    numberOfDecimals: Number(_.product.stockUnit.numberOfDecimals),
                }),
            })) ?? []);
        }
        catch (error) {
            ui.console.error(`Error reading product container:\n${JSON.stringify(error)}`);
            return [];
        }
    }
    return [];
}
/**
 *  Read the product consumption unit
 *  @param pageInstance
 *  @param productCode
 *  @returns
 */
async function getConsumptionUnit(pageInstance, productCode) {
    const _containerUnits = await _getProductContainersUnits(pageInstance, productCode, 10, true);
    if (typesLib.gt(_containerUnits.length, 0)) {
        const _containerUnit = _containerUnits[0];
        return {
            code: _containerUnit.code,
            numberOfConsumptionUnit: _containerUnit.numberOfConsumptionUnit,
            // Number of decimals is only provided if it is from the stock unit and authorized.
            numberOfDecimals: Number(_containerUnit.numberOfDecimals ?? 0),
        };
    }
    return null;
}
/**
 *  Read the product consumption unit Code
 *  @param pageInstance
 *  @param productCode
 *  @returns
 */
async function getConsumptionUnitCode(pageInstance, productCode) {
    return (await getConsumptionUnit(pageInstance, productCode))?.code ?? null;
}
/**
 * Read the product container units list (asynchronous wrapper)
 * @param pageInstance
 * @param productCode
 * @param numberOfRecords
 * @returns
 */
function getProductContainersUnits(pageInstance, productCode, numberOfRecords) {
    return _getProductContainersUnits(pageInstance, productCode, numberOfRecords);
}
//# sourceMappingURL=get-consumption-unit.js.map