"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FifoManagement = void 0;
const typesLib = require("@sage/xtrem-decimal");
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
class FifoManagement {
    constructor(_manufacturedDate, _detentionDate, _sellByDate, _useByDate, _shipByDate, _fifoDate, _errorMessages) {
        this._manufacturedDate = _manufacturedDate;
        this._detentionDate = _detentionDate;
        this._sellByDate = _sellByDate;
        this._useByDate = _useByDate;
        this._shipByDate = _shipByDate;
        this._fifoDate = _fifoDate;
        this._errorMessages = _errorMessages;
    }
    Initialize(_productConfiguration) {
        this._productConfiguration = _productConfiguration;
    }
    // public onValidationManufacturedDate(value: string): string | undefined {
    //     return undefined;
    // }
    async onChangeManufacturedDate(pageInstance) {
        if (this._manufacturedDate.value) {
            this._setStandardDate(this._manufacturedDate.value);
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
    // public onValidationDetentionDate(value: string): string | undefined {
    //     return undefined;
    // }
    async onChangeDetentionDate(pageInstance) {
        if (this._detentionDate.value) {
            this._setManufacturedDate(this._detentionDate.value, 'detentionDate');
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
    onValidationSellByDate(value) {
        if (value &&
            this._detentionDate.value &&
            this._sellByDate.value && typesLib.gte(this._detentionDate.value, this._sellByDate.value)) {
            return this._errorMessages.sellByDateError;
        }
        return undefined;
    }
    async onChangeSellByDate(pageInstance) {
        if (this._sellByDate.value) {
            this._setManufacturedDate(this._sellByDate.value, 'sellByDate');
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
    onValidationUseByDate(value) {
        if (value &&
            this._detentionDate.value &&
            this._useByDate.value && typesLib.gte(this._detentionDate.value, this._useByDate.value)) {
            return this._errorMessages.useByDateError;
        }
        return undefined;
    }
    // eslint-disable-next-line require-await
    async onChangeUseByDate(_pageInstance) {
        if (this._useByDate.value) {
            this._setManufacturedDate(this._useByDate.value, 'useByDate');
        }
    }
    onValidationShipByDate(value) {
        if (value &&
            this._detentionDate.value &&
            this._shipByDate.value && typesLib.gte(this._detentionDate.value, this._shipByDate.value)) {
            return this._errorMessages.shipByDateError;
        }
        return undefined;
    }
    async onChangeShipByDate(pageInstance) {
        if (this._shipByDate.value) {
            this._setManufacturedDate(this._shipByDate.value, 'shipByDate');
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
    // eslint-disable-next-line class-methods-use-this
    onValidationFifoDate(_value) {
        return undefined;
    }
    async onChangeFifoDate(pageInstance) {
        if (this._fifoDate.value) {
            switch (this._productConfiguration?.fifoDateNature) {
                case 'input':
                    // nothing to do in this case
                    break;
                case 'manufacturing':
                    if (!this._manufacturedDate.value) {
                        this._setStandardDate(this._fifoDate.value);
                    }
                    break;
                case 'sellByDate':
                    if (this._sellByDate.value) {
                        this._setManufacturedDate(this._sellByDate.value, 'sellByDate');
                    }
                    break;
                case 'maximumConservationLimit':
                    if (this._useByDate.value) {
                        this._setManufacturedDate(this._useByDate.value, 'useByDate');
                    }
                    break;
                case 'sendByDateLimit':
                    if (this._shipByDate.value) {
                        this._setManufacturedDate(this._shipByDate.value, 'shipByDate');
                    }
                    break;
                default:
                    return;
            }
            await pageInstance.$.commitValueAndPropertyChanges();
        }
    }
    /**
     * This function assign the FIFO date based on the current date and the product configuration.
     * @param currentDate
     * @param fifoDateNature
     * @returns
     */
    _setFifoDate(currentDate, fifoDateNature) {
        if (!this._fifoDate.value && fifoDateNature && typesLib.strictEq(fifoDateNature, this._productConfiguration?.fifoDateNature)) {
            this._fifoDate.value = currentDate;
            return true;
        }
        return false;
    }
    /**
     * This function calculates the manufacturing date and then readjusts the fifo date.
     * @param currentDate
     * @param fieldName
     * @returns
     */
    _setManufacturedDate(currentDate, fieldName) {
        const _currentDate = currentDate ? xtrem_date_time_1.DateValue.parse(currentDate) : undefined;
        if (!this._manufacturedDate.value && _currentDate) {
            const _getManufacturedDate = this._dateCalculation(_currentDate, fieldName, this._productConfiguration);
            if (_getManufacturedDate) {
                this._manufacturedDate.value = _getManufacturedDate;
                this._setFifoDate(_getManufacturedDate, 'manufacturing');
                return this._setStandardDate(_getManufacturedDate);
            }
        }
        return false;
    }
    /**
     * Assign all date after change date
     * @param currentDate
     * @returns
     */
    _setStandardDate(currentDate) {
        const _currentDate = currentDate ? xtrem_date_time_1.DateValue.parse(currentDate) : undefined;
        const { _productConfiguration } = this;
        if (!_productConfiguration || !_currentDate) {
            return false;
        }
        // Define date targets with their corresponding fields and FIFO nature
        const dateTargets = [
            { target: 'detentionDate', field: this._detentionDate, fifoDateNature: undefined },
            { target: 'sellByDate', field: this._sellByDate, fifoDateNature: 'sellByDate' },
            { target: 'useByDate', field: this._useByDate, fifoDateNature: 'maximumConservationLimit' },
            { target: 'shipByDate', field: this._shipByDate, fifoDateNature: 'sendByDateLimit' },
        ];
        // Process each date target using forEach instead of for loop
        dateTargets.forEach(({ target, field, fifoDateNature }) => {
            if (!field.value) {
                // extract date field parameters
                const _dateCalculated = this._dateCalculation(_currentDate, target, _productConfiguration);
                if (_dateCalculated) {
                    // set date value
                    field.value = _dateCalculated;
                    this._setFifoDate(_dateCalculated, fifoDateNature);
                }
            }
        });
        return true;
    }
    /**
     * This function calculates the date based on the current date and the product configuration.
     *  @param currentDate
     *  @param dateTarget
     *  @param productConfiguration
     *  @returns
     */
    // eslint-disable-next-line class-methods-use-this
    _dateCalculation(currentDate, dateTarget, productConfiguration) {
        if (!productConfiguration || !currentDate) {
            return undefined;
        }
        // Define configuration mapping for each date target
        const dateConfigurations = {
            detentionDate: {
                length: productConfiguration.detentionLength,
                unit: productConfiguration.detentionUnit ?? 'no',
            },
            sellByDate: {
                length: productConfiguration.sellByDateLength,
                unit: productConfiguration.sellByDateUnit ?? 'no',
            },
            useByDate: {
                length: productConfiguration.useByDateLength,
                unit: productConfiguration.useByDateUnit ?? 'no',
            },
            shipByDate: {
                length: productConfiguration.sendByDateLength,
                unit: productConfiguration.sendByDateUnit ?? 'no',
            },
        };
        const config = dateConfigurations[dateTarget];
        if (!config) {
            return undefined;
        }
        const { length: _dateLength, unit: _dateUnit } = config;
        // Compute date value based on unit
        switch (_dateUnit) {
            case 'days':
                return currentDate.addDays(_dateLength).toString();
            case 'month':
                return currentDate.addMonths(_dateLength).toString();
            default:
                return currentDate.toString();
        }
    }
}
exports.FifoManagement = FifoManagement;
//# sourceMappingURL=fifo-management.js.map