**# @sage/xtrem-x3-config-manager**

# X3 Config Manager Documentation

This document provides comprehensive documentation for the X3 Config Manager interfaces, their usage patterns, and configuration examples.

## Table of Contents

1. [Overview](#overview)
2. [Core Interfaces](#core-interfaces)
3. [Configuration Structure](#configuration-structure)
4. [Usage Patterns](#usage-patterns)
5. [Environment-Specific Configurations](#environment-specific-configurations)
6. [Security and Encryption](#security-and-encryption)
7. [Best Practices](#best-practices)
8. [Examples](#examples)

## Overview

The X3ConfigManager provides a centralized configuration management system for X3 services, extending the base Xtrem framework configuration with X3-specific settings including database connections, API configurations, SOAP services, and synchronization settings.

### Key Features

- **Multi-environment support**: Development, Azure, and template configurations
- **Security integration**: Syracuse authentication, encryption, and cloud secrets management
- **Database connectivity**: Support for SQL Server (tedious), Oracle, and PostgreSQL drivers
- **Service integration**: API and SOAP web service configurations
- **Cloud support**: AWS SSM and Azure Key Vault integration
- **Synchronization**: Advanced sync configuration for X3 data operations

## Core Interfaces

### 1. X3XtremConfig

The main configuration interface extending the base Xtrem Config.

```typescript
export interface X3XtremConfig extends Config {
    security: X3SecurityConfig;
    x3?: X3Config;
    folderName?: string;
}
```

**Usage**: Root configuration object for all X3 services
**Example**: Used in `X3ConfigManager.current` to access all configuration settings

### 2. X3SecurityConfig

Security configuration extending base SecurityConfig with Syracuse authentication.

```typescript
export interface X3SecurityConfig extends SecurityConfig {
    syracuse?: {
        clientId: string;
        secret: string;
        clockToleranceSeconds?: number;
    };
}
```

**Usage**: Authentication configuration for X3 Syracuse integration
**Security**: Supports SSM parameter store for secrets (`@secret:ssm` prefix)

### 3. X3Config

Core X3-specific configuration containing all service settings.

```typescript
export interface X3Config {
    isCloud: boolean;
    agent?: X3ServiceAgentConfig;
    sql?: X3SqlConfig;
    development?: X3DevelopmentConfig;
    sync?: X3SyncConfig;
}
```

**Usage**: Contains all X3 service configurations including database, development, and sync settings

### 4. X3SqlConfig

Database connection configuration supporting multiple drivers.

```typescript
export interface X3SqlConfig {
    driver: string; // 'tedious', 'oracle', 'postgres'
    hostname: string; // Database server hostname
    port: number; // Database port
    database: string; // Database name/instance
    user: string; // Database username
    password: string; // Database password
    poolSize: number; // Connection pool size
}
```

**Supported Drivers**:

- `tedious`: SQL Server driver
- `oracle`: Oracle database driver
- `postgres`: PostgreSQL driver (implied from framework usage)

### 5. X3DevelopmentConfig

Development environment configuration for local and development setups.

```typescript
export interface X3DevelopmentConfig {
    folderName: string; // X3 folder name (e.g., 'SEED', 'REPOSX3')
    referenceFolder: string; // Reference folder for X3
    defaultLanguage: string; // Default language code (e.g., 'ENG')
    soap: X3SoapConfig; // SOAP web service configuration
    api: X3ApiConfig; // API service configuration
    multiWorkerServiceDisabled?: boolean; // Disable multi-worker for debugging
}
```

### 6. X3ApiConfig

API service configuration for X3 REST API integration.

```typescript
export interface X3ApiConfig {
    url: string; // API endpoint URL
    secret: string; // API secret key
    userCredentials: X3UserCredentialsConfig; // User authentication
    origin?: string; // CORS origin for frontend
}
```

### 7. X3SoapConfig

SOAP web service configuration for X3 SOAP integration.

```typescript
export interface X3SoapConfig {
    webServiceURL: string; // SOAP service URL
    userCredentials: X3UserCredentialsConfig; // User authentication
    codeLang: string; // Language code for SOAP requests
    poolAlias: string; // Connection pool alias
    timeout: number; // Request timeout in milliseconds
}
```

### 8. X3SyncConfig

Synchronization configuration for X3 data sync operations.

```typescript
export interface X3SyncConfig {
    maxPageSize?: number; // Maximum page size for sync operations
    maxStreamBatchSize?: number; // Maximum batch size for streaming
    auth: X3SyncAuthConfig; // Authentication configuration
}
```

### 9. X3SyncAuthConfig

Authentication configuration for sync operations with JWKS and API key support.

```typescript
export interface X3SyncAuthConfig {
    jwksCacheTtlSeconds?: number; // JWKS cache TTL
    ssmCacheTtlSeconds?: number; // SSM cache TTL
    jwksAuthConfig: X3SyncAuthJwksConfig; // JWKS authentication
    x3ApiKeyConfig: X3SyncAuthApiKeyConfig; // API key authentication
}
```

## Configuration Structure

### Environment-Based Configuration

The X3 services support multiple configuration environments:

1. **Development** (`xtrem-config-dev.yml`): Local development with placeholders
2. **Azure** (`xtrem-config-azure.yml`): Azure cloud deployment configuration
3. **Template** (`xtrem-config-template.yml`): Base template for new environments
4. **Main** (`xtrem-config.yml`): Multi-environment configuration with specific setups

### Configuration Hierarchy

```yaml
# Root level
user: ADMIN
deploymentMode: development
storage:
    managedExternal: true

# X3-specific configuration
x3:
    sql: { ... } # Database configuration
    development: { ... } # Development settings
    sync: { ... } # Sync configuration
    agent: { ... } # Service agent settings

# Environment-specific overrides
x3-MGO: { ... } # MongoDB environment
x3-FGR: { ... } # FGR environment
x3-NPO: { ... } # NPO environment
```

## Usage Patterns

### 1. Basic Configuration Access

```typescript
import { X3ConfigManager } from '@sage/xtrem-x3-config-manager';

// Access current configuration
const config = X3ConfigManager.current;
const sqlConfig = config.x3?.sql;
const devConfig = config.x3?.development;
```

### 2. Database Connection Setup

```typescript
const sqlConfig = X3ConfigManager.current.x3?.sql;
if (sqlConfig) {
    const connectionOptions = {
        driver: sqlConfig.driver,
        server: sqlConfig.hostname,
        port: sqlConfig.port,
        database: sqlConfig.database,
        user: sqlConfig.user,
        password: sqlConfig.password,
        pool: {
            max: sqlConfig.poolSize,
            min: 0,
            idleTimeoutMillis: 30000,
        },
    };
}
```

### 3. Syracuse Authentication

```typescript
// Get Syracuse credentials (supports SSM secrets)
const clientId = await X3ConfigManager.getClientId();
const secret = await X3ConfigManager.getSecret();

// Use for authentication
const authConfig = {
    clientId,
    secret,
    clockTolerance: X3ConfigManager.current.security.syracuse?.clockToleranceSeconds ?? 300,
};
```

### 4. Cloud Environment Detection

```typescript
const isCloud = X3ConfigManager.isCloud();
if (isCloud) {
    // Use cloud-specific logic (SSM, Key Vault, etc.)
    const secret = await X3ConfigManager.getSecret();
} else {
    // Use local development configuration
    const secret = X3ConfigManager.current.security.syracuse?.secret;
}
```

### 5. Encryption/Decryption

```typescript
// Encrypt sensitive data
const encryptedData = await X3ConfigManager.encrypt(sensitiveData);

// Decrypt data
const decryptedData = await X3ConfigManager.decrypt(encryptedBase64String);

// Custom secret
const customEncrypted = await X3ConfigManager.encrypt(data, { secret: customSecret });
```

## Environment-Specific Configurations

### Development Environment

```yaml
x3:
    sql:
        driver: tedious
        hostname: <DB_HOST>
        port: <DB_PORT>
        database: <DB_INSTANCE>
        user: <DB_USER>
        password: <DB_USER_PASSWORD>
        poolSize: 5 # Lower for development

    development:
        folderName: <SELECTED_FOLDER>
        referenceFolder: <REFERENCE_FOLDER>
        defaultLanguage: <CURRENT_LANG>
        multiWorkerServiceDisabled: true # Easier debugging

        api:
            url: <API_URL>
            userCredentials:
                userName: admin
                password: admin

        soap:
            webServiceURL: <SOAP_URL>
            userCredentials:
                userName: <SOAP_USERNAME>
                password: <SOAP_PASSWORD>
            codeLang: <CURRENT_LANG>
            poolAlias: <SELECTED_FOLDER>
            timeout: 30000
```

### Azure Cloud Environment

```yaml
x3:
    isCloud: true

    sql:
        driver: tedious
        hostname: your-server.database.windows.net
        port: 1433
        database: X3Database
        user: '@secret:keyvault:/x3/sql/username'
        password: '@secret:keyvault:/x3/sql/password'
        poolSize: 10

    development:
        api:
            url: https://your-x3-api.azurewebsites.net/xtrem/mutation
            secret: '@secret:keyvault:/x3/api/secret'
            origin: https://your-app.azurefd.net
            userCredentials:
                userName: '@secret:keyvault:/x3/api/username'
                password: '@secret:keyvault:/x3/api/password'

    sync:
        maxPageSize: 2000
        maxStreamBatchSize: 200
        auth:
            jwksCacheTtlSeconds: 3600
            ssmCacheTtlSeconds: 300
            jwksAuthConfig:
                audience: api://your-azure-ad-app-id
                tokenUri: https://login.microsoftonline.com/your-tenant-id/discovery/v2.0/keys
                cloudDomain: your-domain.onmicrosoft.com
            x3ApiKeyConfig:
                awsSsmParamPath: /azure/x3/api/key
                xApiKey: '@secret:keyvault:/x3/api/key'
```

### Multi-Environment Setup

```yaml
# Multiple X3 environments in single config
x3-MGO: # MongoDB environment
    sql:
        driver: oracle
        hostname: scmx3-dev-mgo.sagex3rd.local
        port: 1521
        database: xe.oracle.docker
        user: REPOSX3
        password: tiger

x3-FGR: # FGR environment
    sql:
        driver: oracle
        hostname: scmx3-dev-fgr.sagex3rd.local
        port: 1521
        database: xe.oracle.docker
        user: REPOSX3
        password: tiger

x3: # Unit tests environment
    sql:
        driver: tedious
        hostname: x3-adcut.sagex3rd.local\SAGEX3,1433
        database: x3
        user: X3RBTRUN
        password: Manager74
```

## Security and Encryption

### 1. Secret Management

**SSM Parameter Store (AWS)**:

```yaml
security:
    syracuse:
        clientId: '@secret:ssm:/x3/syracuse/clientId'
        secret: '@secret:ssm:/x3/syracuse/secret'
```

**Azure Key Vault**:

```yaml
api:
    secret: '@secret:keyvault:/x3/api/secret'
    userCredentials:
        userName: '@secret:keyvault:/x3/api/username'
        password: '@secret:keyvault:/x3/api/password'
```

### 2. Encryption Usage

```typescript
// Encrypt user data before storage
const userData = { username: 'admin', permissions: ['read', 'write'] };
const encrypted = await X3ConfigManager.encrypt(userData);
// Store encrypted.iv and encrypted.encrypted

// Decrypt when needed
const decrypted = await X3ConfigManager.decrypt(Buffer.from(JSON.stringify(encrypted)).toString('base64'));
```

### 3. Validation

The config manager includes built-in validation:

```typescript
// Validates sync auth configuration
X3ConfigManager.validateX3SyncConfig(config.x3.sync.auth);

// Throws errors for missing required properties:
// - jwksAuthConfig.audience
// - jwksAuthConfig.tokenUri
// - jwksAuthConfig.cloudDomain
// - x3ApiKeyConfig.awsSsmParamPath
// - x3ApiKeyConfig.xApiKey
```

## Best Practices

### 1. Environment Configuration

- **Use placeholders** in development configs (`<DB_HOST>`, `<API_URL>`)
- **Separate environments** with prefixed keys (`x3-DEV`, `x3-PROD`)
- **Enable debugging features** in development (`multiWorkerServiceDisabled: true`)
- **Optimize pool sizes** per environment (5 for dev, 10+ for prod)

### 2. Security

- **Never commit secrets** to version control
- **Use secret management** systems (SSM, Key Vault) for production
- **Validate configurations** on startup
- **Use encryption** for sensitive data storage
- **Implement proper CORS** origins for APIs

### 3. Performance

- **Configure appropriate timeouts** (30s for dev, 50s for prod)
- **Set optimal pool sizes** based on expected load
- **Cache TTL settings** for auth tokens (3600s for JWKS, 300s for SSM)
- **Batch size configuration** for sync operations

### 4. Development

- **Use local override configs** for developer-specific settings
- **Disable multi-worker services** for easier debugging
- **Set verbose logging** for development environments
- **Use shorter timeouts** for faster feedback

## Examples

### Complete Development Setup

```typescript
// Load configuration
const config = X3ConfigManager.load('./config');

// Setup database connection
const sqlConfig = config.x3?.sql;
if (!sqlConfig) {
    throw new Error('SQL configuration not found');
}

// Setup API client
const apiConfig = config.x3?.development?.api;
if (!apiConfig) {
    throw new Error('API configuration not found');
}

// Setup SOAP client
const soapConfig = config.x3?.development?.soap;
if (!soapConfig) {
    throw new Error('SOAP configuration not found');
}

console.log(`Connected to X3 folder: ${config.x3?.development?.folderName}`);
console.log(`Using language: ${config.x3?.development?.defaultLanguage}`);
console.log(`Database driver: ${sqlConfig.driver}`);
```

### Cloud Production Setup

```typescript
// Load configuration with validation
const config = X3ConfigManager.load('./config');

// Detect cloud environment
if (X3ConfigManager.isCloud()) {
    console.log('Running in cloud environment');

    // Get credentials from secret store
    const clientId = await X3ConfigManager.getClientId();
    const secret = await X3ConfigManager.getSecret();

    // Setup with cloud credentials
    setupSyracuseAuth(clientId, secret);
} else {
    console.log('Running in local environment');
}

// Validate sync configuration if present
if (config.x3?.sync?.auth) {
    X3ConfigManager.validateX3SyncConfig(config.x3.sync.auth);
    console.log('Sync configuration validated successfully');
}
```

### Multi-Environment Service

```typescript
class X3EnvironmentManager {
    private configs: Map<string, X3XtremConfig> = new Map();

    async loadEnvironment(envName: string): Promise<void> {
        const config = X3ConfigManager.load(`./config-${envName}`);
        this.configs.set(envName, config);

        // Validate environment-specific requirements
        if (config.x3?.sql) {
            await this.validateDatabaseConnection(config.x3.sql);
        }

        if (config.x3?.development?.api) {
            await this.validateApiConnection(config.x3.development.api);
        }
    }

    getEnvironment(envName: string): X3XtremConfig {
        const config = this.configs.get(envName);
        if (!config) {
            throw new Error(`Environment ${envName} not loaded`);
        }
        return config;
    }

    private async validateDatabaseConnection(sqlConfig: X3SqlConfig): Promise<void> {
        // Validate database connection
        console.log(`Validating connection to ${sqlConfig.hostname}:${sqlConfig.port}`);
        // Implementation would test actual connection
    }

    private async validateApiConnection(apiConfig: X3ApiConfig): Promise<void> {
        // Validate API connection
        console.log(`Validating API connection to ${apiConfig.url}`);
        // Implementation would test actual API endpoint
    }
}
```

This documentation provides a comprehensive guide to understanding and using the X3 Config Manager interfaces and their practical applications in X3 service development.
